from __future__ import unicode_literals

import errno
import os

from its_client import poller
from sepelib.util import fs

from instancectl import constants


class ItsVolumePlugin(object):

    def __init__(self, service_id, auto_tags, first_poll_timeout, shared_storage, controls_real_path):
        """
        :type service_id: unicode
        :type auto_tags: list[unicode]
        :type shared_storage: unicode
        :type first_poll_timeout: int
        :type controls_real_path: unicode
        """
        self.service_id = service_id
        self.auto_tags = auto_tags
        self.first_poll_timeout = first_poll_timeout
        self.shared_storage = shared_storage
        self.controls_real_path = controls_real_path
        self.poller = None

    @staticmethod
    def _make_poller(volume, service_id, auto_tags, shared_storage, controls_work_dir_path):
        i = volume.its_volume
        return poller.ItsPoller(
            url=i.its_url,
            service_id=service_id,
            itags=auto_tags,
            controls_dir=controls_work_dir_path,
            shared_dir=shared_storage,
            refresh_flag=constants.ITS_REFRESH_FLAG_FILENAME,
            req_timeout=constants.ITS_REQUEST_TIMEOUT,
            poll_timeout=i.period_seconds,
            max_poll_timeout=i.max_retry_period_seconds,
            max_timeout_jitter=constants.ITS_PERIOD_JITTER,
        )

    def _create_symlink_ignore(self, src, dst):
        try:
            os.symlink(src, dst)
        except OSError as e:
            if e.errno != errno.EEXIST:
                raise
            if os.readlink(dst) != src:
                raise

    def setup(self, work_dir, v):
        """
        :type work_dir: unicode
        :type v: types_pb2.Volume
        """
        controls_work_dir_path = os.path.join(work_dir, v.name)
        if self.controls_real_path:
            fs.makedirs_ignore(self.controls_real_path)
            self._create_symlink_ignore(self.controls_real_path, controls_work_dir_path)
        self.poller = self._make_poller(volume=v,
                                        service_id=self.service_id,
                                        auto_tags=self.auto_tags,
                                        shared_storage=self.shared_storage,
                                        controls_work_dir_path=controls_work_dir_path)
        self.poller.start()
        self.poller.controls_requested_event.wait(self.first_poll_timeout)

    def stop(self):
        if self.poller is not None:
            self.poller.stop()
