from __future__ import unicode_literals

import os

import jinja2
from sepelib.util.fs import makedirs_ignore

from instancectl.hq.volumes import template_filters
from instancectl.hq.volumes import errors
from instancectl import utils


class TemplateVolumePlugin(object):
    def __init__(self, context, instance_dir):
        self.context = context
        self.env = jinja2.Environment(loader=jinja2.FileSystemLoader([instance_dir, '/']),
                                      undefined=jinja2.StrictUndefined)
        self._add_filters_to_env()

    def setup(self, work_dir, volume):
        """
        :type work_dir: unicode
        :type volume: types_pb2.Volume
        """
        p = os.path.join(work_dir, volume.name)
        for template in volume.template_volume.template:
            self._process_template(p, template, self.context)

    def _process_template(self, volume_path, template, context):
        """
        :type volume_path: unicode
        :type template: clusterpb.types_pb2.TemplateVolume
        :type context: dict
        """
        dst_path = template.dst_path
        try:
            content = self._render_content(template.src_path, context)
        except Exception as e:
            raise utils.make_exc(errors.TemplateProcessingError, 'Cannot process template {}: {}'.format(dst_path, e))
        try:
            self._write_rendered_content(volume_path, dst_path, content)
        except Exception as e:
            raise utils.make_exc(errors.TemplateProcessingError,
                                 'Cannot save volume "{}" render result to "{}": {}'.format(volume_path, dst_path, e))

    @staticmethod
    def _write_rendered_content(dir_name, file_name, content):
        """
        :type dir_name: unicode
        :type file_name: unicode
        :type content: unicode
        """
        path = os.path.join(dir_name, file_name)
        makedirs_ignore(dir_name)
        with open(path, 'w') as template_dst:
            if isinstance(content, unicode):
                content = content.encode('utf-8')
            template_dst.write(content)

    def _render_content(self, src_path, context):
            template = self.env.get_template(src_path)
            return template.render(**context)

    def _add_filters_to_env(self):
        self.env.filters.update({
            'last_revision': template_filters.last_revision,
            'alive': template_filters.alive,
            'alive_or_last_revision': template_filters.alive_or_last_revision,
            'revision_filter': template_filters.revision_filter,

            'ready_endpoints': template_filters.ready_endpoints
        })
