from __future__ import unicode_literals

import inject
import gevent
import gevent.event
import gevent.pool
import gevent.timeout

from instancectl import constants
from . import helpers


class IJobController(object):

    @classmethod
    def instance(cls):
        """
        :rtype: IJobController
        """
        return inject.instance(cls)

    def stop(self):
        raise NotImplementedError

    def reopenlogs(self):
        raise NotImplementedError

    def notify(self, updates):
        """
        :type updates: list[unicode]
        """
        raise NotImplementedError

    def uninstall_instance(self):
        raise NotImplementedError

    def init_containers_streamfiles(self, env):
        """
        Returns init containers stdout & stderr files

        :type env: instancectl.lib.envutil.InstanceCtlEnv

        :rtype: list[str]
        """
        raise NotImplementedError


class JobController(IJobController):

    def __init__(self, job_pool, action_stop_timeout):
        """
        :type job_pool: instancectl.jobs.jobpool.JobPool
        :type action_stop_timeout: int | float
        """
        self._action_stop_timeout = action_stop_timeout
        self.job_pool = job_pool

    def start(self):
        helpers.init_crash_log(constants.CRASH_LOG_PATH)
        self.job_pool.prepare_instance()
        self.job_pool.start()

    def wait(self):
        self.job_pool.wait()

    def stop(self):
        result = False
        with gevent.Timeout(self._action_stop_timeout, False):
            self.job_pool.stop()
            result = True
        return result

    def reopenlogs(self):
        self.job_pool.reopenlogs()

    def notify(self, updates):
        return self.job_pool.notify(updates)

    def prepare_instance(self):
        self.job_pool.prepare_instance()

    def uninstall_instance(self):
        self.job_pool.uninstall_instance()

    def all_jobs_running(self):
        """
        :rtype: bool
        """
        for j in self.job_pool.jobs.itervalues():
            if j.ready():
                return False
        return True

    def init_containers_streamfiles(self, env):
        """
        Returns init containers stdout & stderr files

        :type env: instancectl.lib.envutil.InstanceCtlEnv

        :rtype: list[str]
        """
        return self.job_pool.init_containers_streamfiles(env)
