from __future__ import unicode_literals

import resource
import logging
import logging.handlers
import os

from sepelib.util import fs
from sepelib.util.log import formatters

from instancectl import common
from instancectl.config.defaults import run_limit_types
from . import errors

LIMIT_UNLIMITED = 'unlimited'


def init_crash_log(path):
    """
    :type path: unicode
    :rtype: logging.Logger
    """
    d = os.path.dirname(path)
    fs.makedirs_ignore(d)
    logger = logging.getLogger('crash')
    handler = logging.handlers.RotatingFileHandler(filename=path, maxBytes=10 ** 5, backupCount=15)
    # format is %(message)s by default
    formatter = formatters.TabSeparatedKeyValueFormatter()
    handler.setFormatter(formatter)
    logger.addHandler(handler)
    return logger


def make_limit(limit_id, value):
    hard_limit = resource.getrlimit(limit_id)[1]

    if value == LIMIT_UNLIMITED:
        soft_limit = hard_limit
    else:
        try:
            value = int(value)
        except ValueError:
            raise errors.LimitValueError('Incorrect limit value: {}'.format(value))

        if hard_limit == resource.RLIM_INFINITY:
            soft_limit = value
        else:
            soft_limit = min(value, hard_limit)

    return common.RLimit(soft_limit, hard_limit)


def make_limits_dict_from_config(run_config):
    """
    Makes limits dict from job config.

    :type run_config: dict
    :rtype: dict[unicode, instancectl.common.RLimit]
    """
    result = {}
    for key, limit_id in run_limit_types.iteritems():
        value = run_config.get('limit_{}'.format(key))
        if not value and value != 0:
            continue
        result[key] = make_limit(limit_id, value)
    return result

