from __future__ import unicode_literals

import logging
import logging.handlers
import os
import shutil
import sys

import gevent
import gevent.event
import gevent.pool
import gevent.queue
import gevent.lock

import six

from sepelib.gevent import greenthread
from sepelib.util import fs

from yp_proto.yp.client.hq.proto import types_pb2

from instancectl.utils import get_event_logger, tryopen
from instancectl.lib import handler
from instancectl.lib import specutil
from instancectl import constants
from instancectl import errors
from instancectl import cms


class JobPool(greenthread.GreenThread):
    """
    :type _handler_runner: instancectl.lib.handler.HandlerRunner
    """

    ITERATION_DELAY = 0.5

    def __init__(self, jobs, spec, secret_volume_plugin, template_volume_plugin, vault_client, yav_client,
                 init_containers, status_cacher, its_volume_plugin, env):
        """
        :type jobs: dict[unicode, instancectl.jobs.job.Job]
        :type spec: clusterpb.types_pb2.InstanceRevision
        :type secret_volume_plugin: instancectl.hq.volumes.secret.SecretVolumePlugin
        :type template_volume_plugin: instancectl.hq.volumes.template.TemplateVolumePlugin
        :type vault_client: instancectl.clients.vault.client.VaultClient
        :type yav_client: instancectl.clients.yav.client.YavClient
        :type init_containers: list[instancectl.jobs.job.Job]
        :type status_cacher: instancectl.status.cacher.InstanceRevisionStatusCacher
        :type its_volume_plugin: instancectl.hq.volumes.its.ItsVolumePlugin
        :type env: instancectl.lib.envutil.InstanceCtlEnv
        """
        super(JobPool, self).__init__()
        self.log = logging.getLogger(__name__ + '.' + self.__class__.__name__)
        self._event_log = get_event_logger()
        self.log.debug('Initializing JobPool')
        self.jobs = jobs
        self.run_flag_path = os.path.abspath(constants.RUN_FLAG_PATH)
        # For garbage collector
        self.runflag_fd = None
        self._secret_volume_plugin = secret_volume_plugin
        self._template_volume_plugin = template_volume_plugin
        self._its_volume_plugin = its_volume_plugin
        self._spec = spec
        self._vault_client = vault_client
        self._yav_client = yav_client
        self._init_containers = init_containers
        self._status_cacher = status_cacher
        self._install_flag = os.path.join(constants.STATE_DIR, 'install.flag')
        self._notify_result = None
        self._notify_mutex = gevent.lock.Semaphore()
        self._term_barrier = gevent.event.Event()
        self._instancectl_env = env
        self._handler_runner = None

    def _is_installed(self):
        """
        :rtype: bool
        """
        return os.path.exists(self._install_flag)

    @staticmethod
    def _is_init_container_processed(name):
        """
        :type name: unicode
        :rtype: bool
        """
        p = os.path.join(constants.STATE_DIR, '{}_install.flag'.format(name))
        return os.path.exists(p)

    @staticmethod
    def set_init_container_processed(name):
        p = os.path.join(constants.STATE_DIR, '{}_install.flag'.format(name))
        open(p, 'w').close()

    def _set_installed_condition(self, s):
        cond = types_pb2.Condition()
        cond.last_transition_time.GetCurrentTime()
        if s:
            cond.status = constants.CONDITION_TRUE
            cond.reason = 'RevisionInstalled'
        else:
            cond.status = constants.CONDITION_FALSE
            cond.reason = 'RevisionNotInstalled'
        self._status_cacher.set_installed_condition(cond)

    def prepare_instance(self):
        try:
            self._prepare_instance()
        # Catch gevent.Timeout here because it is inherited from BaseException:
        # https://st.yandex-team.ru/SWAT-7840#61964a04c3202303d802e342.
        except (gevent.Timeout, Exception) as e:
            self.log.exception('Cannot prepare instance')
            self._set_installed_condition(False)
            raise errors.InstanceCtlInitError('Cannot prepare instance: {}'.format(e)), None, sys.exc_info()[2]

    def _make_resources_symlinks(self):
        # It's WRONG way of resources symlinking, but we have to use it without https://st.yandex-team.ru/ISS-4842
        c = cms.load_dump_json()
        for r in c['resources']:
            if r in constants.SYMLINKING_RESOURCES_BLACKLIST:
                continue
            src = os.path.abspath(r)
            dst = os.path.join(self._spec.work_dir, r)
            self.log.debug('Creating symlink for %s -> %s', src, dst)
            os.symlink(src, dst)

    def _prepare_instance(self):
        """
        Prepares instance.
        """
        # Instance may be installed already. So we MUST check if instance
        # installed before the first report to HQ.
        self._event_log = get_event_logger('install')
        self.log.debug('Installing instance')
        fs.makedirs_ignore(constants.STATE_DIR)
        self._open_run_flag()
        self._process_hq_spec()
        if self._spec.work_dir:
            fs.makedirs_ignore(self._spec.work_dir)
        for j in six.itervalues(self.jobs):
            j.init()
        self._handler_runner = handler.HandlerRunner(
            http_get_runner=handler.HttpGetRunner(),
            exec_runner=handler.ExecRunner(
                stdout=open('notify.out', 'a'),
                stderr=open('notify.err', 'a'),
                env=self._instancectl_env.default_container_env,
                limits={},
                pass_fds=[],
                log=logging.getLogger('notify-action')
            )
        )
        if self._is_installed():
            self.log.debug('Instance installed already')
            self._set_installed_condition(True)
            return
        if self._spec.work_dir:
            self._make_resources_symlinks()
        self._set_installed_condition(False)
        for c in self._init_containers:
            if not self._is_init_container_processed(c.name):
                c.set_run_flag_fd(self.runflag_fd)
                c.run()
                self.set_init_container_processed(c.name)
        open(self._install_flag, 'w').close()
        self._set_installed_condition(True)

    def all_jobs_running(self):
        """
        Returns True if all threads watching for binaries are alive.

        :rtype: bool
        """
        for section, job in six.iteritems(self.jobs):
            if job.ready():
                self.log.error('Section %s thread crashed! Stopping instance', section)
                self._event_log.error('Section %s thread crashed! Stopping instance', section)
                return False
        return True

    def run(self):
        self._event_log = get_event_logger('start')
        self.log.debug('Starting instance')
        try:
            for job in six.itervalues(self.jobs):
                job.set_run_flag_fd(self.runflag_fd)
                job.start()
            while True:
                # bsconfig stops instances via run.flag removing
                # we must check if it exists
                if not os.path.exists(self.run_flag_path):
                    self.log.info('run.flag disappeared. Stopping instance')
                    self._event_log.info('run.flag disappeared. Stopping instance')
                    break
                if not self.all_jobs_running():
                    break
                gevent.sleep(self.ITERATION_DELAY)
        finally:
            self.stop_all()
            try:
                shutil.rmtree(constants.PIDS_DIR, ignore_errors=True)
            except Exception:
                self.log.exception('Failed removing pids directory')

    def _open_run_flag(self):
        # We must open run.flag.
        # All subprocesses will inherit this open descriptor, so
        # bsconfig will be able to run fuser on it and kill whole subprocess tree
        self.runflag_fd = tryopen(self.run_flag_path, 'a')
        if self.runflag_fd is None:
            self.log.critical('Failed opening run.flag!')
            self._event_log.critical('Failed opening run.flag!')
            raise errors.InstanceCtlInitError('Cannot open run.flag file, which must be opened before start')

    def _process_hq_spec(self):
        # Here we try to process spec from HQ
        # we MUST NOT start instance before we do it
        for v in self._spec.volume:
            self.log.info('Setup volume %s', v.name)
            if v.type == types_pb2.Volume.SECRET:
                self._secret_volume_plugin.setup(self._spec.work_dir,
                                                 client=self._vault_client,
                                                 volume_name=v.name,
                                                 secret=v.secret_volume.keychain_secret)
            elif v.type == types_pb2.Volume.VAULT_SECRET:
                self._secret_volume_plugin.setup(self._spec.work_dir,
                                                 client=self._yav_client,
                                                 volume_name=v.name,
                                                 secret=v.vault_secret_volume.vault_secret)
            elif v.type == types_pb2.Volume.TEMPLATE:
                self._template_volume_plugin.setup(self._spec.work_dir, v)
            elif v.type == types_pb2.Volume.ITS:
                # We MUST NOT start instances before the first attempt of ITS polling (successful or not)
                self._its_volume_plugin.setup(self._spec.work_dir, v)
        for name, job in six.iteritems(self.jobs):
            self.log.info('Process container "%s" settings', name)
            job.process_spec(self._vault_client, self._yav_client)
        for job in self._init_containers:
            self.log.info('Process init container "%s" settings', job.name)
            job.process_spec(self._vault_client, self._yav_client)
        if self._instancectl_env.use_spec:
            specutil.expand_revision_spec_variables(self._spec, self._instancectl_env.default_container_env)

    def stop_all(self):
        self.log.info('Stop all jobs')
        self._event_log.info('Stop all jobs')

        thread_group = gevent.pool.Group()
        for job in six.itervalues(self.jobs):
            t = gevent.spawn(job.stop_container, self._term_barrier)
            thread_group.add(t)
        self.log.info('Waiting for jobs to be stopped')
        for job in six.itervalues(self.jobs):
            job.pre_stop_action_finished.wait()
        self._term_barrier.set()
        thread_group.join()
        self.log.info('All jobs have been stopped')

    def reopenlogs(self):
        for job in six.itervalues(self.jobs):
            job.reopenlogs()

    def uninstall_instance(self):
        """
        Makes actions before the instance removal.
        """
        for job in six.itervalues(self.jobs):
            job.init()
            job.uninstall()

    def notify(self, updates):
        """
        :type updates: list[unicode]
        """
        if not self._notify_result:
            self._notify_result = gevent.event.AsyncResult()
            gevent.spawn(self._notify, updates)
        self.log.info('Waiting for notify action to be finished')
        return self._notify_result.get()

    def _notify(self, updates):
        """
        Runs notify_action.

        Script arguments are equal to iss_hook_notify arguments.

        To make it compatible with iss_hook_notify we add string
        "notify_script" as first argument, to make real arguments
        be indexed from 1 as it was in iss_hook_notify.

        :type updates: list[unicode]
        """
        with self._notify_mutex:
            self.log.info('Running notify actions of all sections')
            r = self._notify_result
            self._notify_result = None
            for i, h in enumerate(self._spec.notify_action.handlers):
                self.log.info('Running notify action %s', i)
                args = handler.HandlerRunnerArguments(exec_args=['notify_action'] + updates)
                try:
                    self._handler_runner.run(h, args)
                except handler.HandlerRunnerError as e:
                    self.log.warning('Notify action failed')
                    c = types_pb2.Condition()
                    c.status = 'False'
                    c.reason = 'NotifyActionFailed'
                    c.message = 'Notify action failed: {}'.format(e)
                    c.last_transition_time.GetCurrentTime()
                    r.set(c)
                    return
            c = types_pb2.Condition()
            c.status = 'True'
            c.reason = 'NotifyActionOk'
            c.last_transition_time.GetCurrentTime()
            r.set(c)

    def init_containers_streamfiles(self, env):
        """
        Returns init containers stdout & stderr files

        :type env: instancectl.lib.envutil.InstanceCtlEnv

        :rtype: list[str]
        """
        filenames = []
        for c in self._init_containers:
            stdout_path = os.path.join(env.instance_dir, '{}.out'.format(c.name))
            stderr_path = os.path.join(env.instance_dir, '{}.err'.format(c.name))
            filenames.append(stdout_path)
            filenames.append(stderr_path)
        if self._spec.notify_action.handlers:
            notify_stdout_path = os.path.join(env.instance_dir, 'notify.out')
            notify_stderr_path = os.path.join(env.instance_dir, 'notify.err')
            filenames.append(notify_stdout_path)
            filenames.append(notify_stderr_path)
        return filenames
