from __future__ import unicode_literals

import logging
import os
import stat

import fallocate
import gevent
from sepelib.gevent import greenthread

from instancectl import constants


log = logging.getLogger('stdout-rotater')


class StdoutRotater(greenthread.GreenThread):

    LOOP_DELAY = 30.0

    # Must use constant 512:
    # blkcnt_t  st_blocks;      /* Number of 512B blocks allocated */
    BLOCK_SIZE = 512

    def __init__(self, filenames, size_limit):
        """
        :type filenames: list
        :type size_limit: int
        """
        self._filenames = filenames
        self._size_limit = size_limit
        super(StdoutRotater, self).__init__()

    def _rotate_if_needed(self, filename):
        """
        :type filename: unicode
        """
        # Copied from:
        # https://github.com/yandex/porto/blob/9c70cba7c6222392be8f3b3302ded4e388a20d3b/src/util/path.cpp#L561
        # See also: http://man7.org/linux/man-pages/man2/fallocate.2.html
        with open(filename, 'r+') as f:
            fd = f.fileno()
            st = os.fstat(fd)

            if not stat.S_ISREG(st.st_mode):
                return

            if st.st_blocks * self.BLOCK_SIZE <= self._size_limit:
                return

            hole_len = st.st_size - self._size_limit / 2
            hole_len -= hole_len % st.st_blksize

            try:
                fallocate.fallocate(fd, 0, hole_len, constants.FALLOC_FL_COLLAPSE_RANGE)
            except Exception as e:
                log.warning('Cannot rotate file "%s": fallocate fails: %s', filename, e)
            else:
                return
            os.ftruncate(fd, 0)

    def _run(self):
        for f in self._filenames:
            try:
                self._rotate_if_needed(f)
            except Exception as e:
                log.exception('Cannot rotate file "%s": %s', f, e)

    def run(self):
        while True:
            try:
                self._run()
            except Exception:
                log.exception('Stdout/stderr truncater thread fail')
            gevent.sleep(self.LOOP_DELAY)
