from __future__ import unicode_literals

import collections
import os
import os.path
from yp_proto.yp.client.hq.proto import types_pb2

from instancectl import constants
from instancectl import hq
from instancectl.actions import install
from instancectl.actions import start
from instancectl.actions import uninstall
from instancectl.clients.vault import client as vault
from instancectl.clients.yav import client as yav
from instancectl.config import config
from instancectl.hq.volumes import its
from instancectl.hq.volumes import secret
from instancectl.hq.volumes import template
from instancectl.jobs import controller
from instancectl.jobs import jobpool
from instancectl.jobs import stdout_rotater
from instancectl.lib import confutil
from instancectl.lib import envutil
from instancectl.lib import fileutil
from instancectl.lib import specutil
from instancectl.status import cacher
from instancectl.status.updater import container
from instancectl.status.updater import instance as instance_updater


def make_status_cacher_from_spec(spec):
    """
    :type spec: clusterpb.types_pb2.InstanceRevision
    """
    s = types_pb2.RevisionStatus()
    s.installed.status = constants.CONDITION_FALSE
    s.installed.reason = 'RevisionNotInstalled'
    s.installed.last_transition_time.GetCurrentTime()
    s.ready.status = constants.CONDITION_FALSE
    s.ready.reason = 'RevisionNotReady'
    s.ready.last_transition_time.GetCurrentTime()
    criterion_containers = []
    for c in spec.container:
        cont = s.container.add()
        cont.name = c.name
        cont.ready.status = constants.CONDITION_FALSE
        cont.ready.reason = 'ContainerNotReady'
        cont.ready.last_transition_time.GetCurrentTime()
        cont.installed.status = constants.CONDITION_FALSE
        cont.installed.reason = 'ContainerNotInstalled'
        cont.installed.last_transition_time.GetCurrentTime()
        criterion_containers.append(c.name)
    return cacher.InstanceRevisionStatusCacher(status=s, criterion_containers=criterion_containers)


def make_status_updater_from_spec(job_ctrl, status_cacher):
    updaters = []
    for j in job_ctrl.job_pool.jobs.itervalues():
        probe = j.spec.readiness_probe
        u = container.ContainerStatusUpdater(job=j, spec=probe, cacher=status_cacher)
        updaters.append(u)
    return instance_updater.InstanceRevisionStatusUpdater(updaters)


def make_hq_status_reporter(env, status_cacher, hq_client, instance_id):
    if env.hq_report_version == 1:
        report_executor = hq.reporter.ReportExecutorV1(hq_client)
    elif env.hq_report_version == 2:
        report_executor = hq.reporter.ReportExecutorV2(hq_client)
    else:
        raise ValueError("env.hq_report_version %s is not valid, available (1, 2) only".format(env.hq_report_version))

    return hq.reporter.HqInstanceStatusReporter(
        report_executor=report_executor,
        status_cacher=status_cacher,
        min_report_delay=constants.HQ_MIN_REPORT_DELAY,
        max_report_delay=constants.HQ_MAX_REPORT_DELAY,
        max_report_delay_jitter=constants.HQ_MAX_REPORT_DELAY_JITTER,
        instance_id=instance_id,
        enable_hq_report=env.hq_report,
    )


def make_job_ctrl_from_spec(spec, env, status_cacher):
    """
    :type spec: clusterpb.types_pb2.InstanceRevision
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: instancectl.jobs.controller.JobController
    """
    vault_client = vault.VaultClient(constants.DEFAULT_VAULT_URL, service_id=env.service_id)
    yav_client = yav.YavClient(constants.DEFAULT_YAV_URL,
                               service_id=env.service_id,
                               tvm_client_id=constants.DEFAULT_TVM_CLIENT_ID,
                               yav_tvm_client_id=constants.DEFAULT_YAV_TVM_CLIENT_ID,
                               tvm_api_url=constants.DEFAULT_TVM_API_URL)

    # Instance volume plugins
    context = confutil.make_template_plugin_context(constants.DEFAULT_FEDERATED_URL, env)
    template_volume_plugin = template.TemplateVolumePlugin(context, instance_dir=env.instance_dir)
    secret_volume_plugin = secret.SecretVolumePlugin()

    controls_real_path = ''
    for volume in spec.volume:
        if volume.type == volume.ITS and volume.its_volume.use_shared_memory:
            controls_real_path = os.path.join('/dev/shm', env.instance_id)

    its_volume_plugin = its.ItsVolumePlugin(service_id=env.service_id,
                                            auto_tags=env.auto_tags,
                                            first_poll_timeout=constants.ITS_FIRST_POLL_TIMEOUT,
                                            shared_storage=envutil.make_its_shared_storage_dir(env),
                                            controls_real_path=controls_real_path)

    jobs = collections.OrderedDict()
    for c in spec.container:
        sleeper = specutil.create_restart_sleeper(c)
        jobs[c.name] = confutil.make_job_from_spec(
            c, env, spec, sleeper, constants.DEFAULT_CONTAINER_RESTART_SUCCESSFUL_START_TIMEOUT
        )
    init_containers = []
    for c in spec.init_containers:
        sleeper = envutil.create_restart_sleeper(env.prepare_script_restart_policy)
        cont = confutil.make_job_from_spec(c, env, spec, sleeper=sleeper, successful_start_timeout=None)
        init_containers.append(cont)
    job_pool = jobpool.JobPool(
        jobs=jobs,
        spec=spec,
        init_containers=init_containers,
        secret_volume_plugin=secret_volume_plugin,
        template_volume_plugin=template_volume_plugin,
        vault_client=vault_client,
        yav_client=yav_client,
        status_cacher=status_cacher,
        its_volume_plugin=its_volume_plugin,
        env=env,
    )
    return controller.JobController(job_pool=job_pool,
                                    action_stop_timeout=constants.DEFAULT_INSTANCE_SPEC_STOP_TIMEOUT)


def make_stdout_rotater(job_ctrl, env):
    """
    :type job_ctrl: instancectl.jobs.controller.JobController
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: instancectl.jobs.stdout_rotater.StdoutRotater
    """
    filenames = []
    for n, j in job_ctrl.job_pool.jobs.iteritems():
        # Porto rotates stdout and stderr files of container root process,
        # so we have to rotate stdout/stderr only for sections starting via fork && exec
        # https://github.com/yandex/porto/blob/master/porto.md#filesystem
        if not j.porto_mode.enabled:
            stdout_path = os.path.join(env.instance_dir, '{}.out'.format(n))
            stderr_path = os.path.join(env.instance_dir, '{}.err'.format(n))
            filenames.append(stdout_path)
            filenames.append(stderr_path)
    filenames.extend(job_ctrl.init_containers_streamfiles(env))
    return stdout_rotater.StdoutRotater(
        filenames=filenames,
        size_limit=constants.STDOUT_STDERR_SIZE_LIMIT_BYTES,
    )


def get_instance_spec(hq_client, env):
    """
    :type hq_client: instancectl.hq.instance_client.HqInstanceClient
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: clusterpb.types_pb2.InstanceRevision
    """
    if env.yp_hq_spec is not None:
        s = env.yp_hq_spec
    elif env.use_spec or env.hq_poll:
        s = hq_client.get_instance_revision(env.instance_id)
    else:
        s = types_pb2.InstanceRevision()
    return specutil.prepare_instance_spec(s, env)


def make_install_action(config_path, hq_url):
    """
    :type config_path: unicode
    :type hq_url: unicode | types.NoneType
    :rtype: instancectl.actions.install.InstallAction
    """
    env = envutil.make_instance_ctl_env(hq_url)
    hq_client = hq.instance_client.HqInstanceClient(env.hq_url)
    spec = get_instance_spec(hq_client, env)
    if env.use_spec:
        status_cacher = make_status_cacher_from_spec(spec)
        job_ctrl = make_job_ctrl_from_spec(spec, env, status_cacher)
    else:
        conf = config.get_or_compile_instance_config(config_path, env)
        status_cacher = confutil.make_status_cacher_from_config(conf)
        job_ctrl = confutil.make_job_ctrl_from_config(conf, spec, env, status_cacher)
    hq_status_reporter = make_hq_status_reporter(env, status_cacher, hq_client, env.instance_id)
    return install.InstallAction(job_ctrl, hq_status_reporter)


def make_start_action(config_path, hq_url):
    """
    :type config_path: unicode
    :type hq_url: unicode | types.NoneType
    :rtype: instancectl.application.Application
    """
    # We must acquire file lock as soon as possible
    lock = fileutil.lock_file()
    env = envutil.make_instance_ctl_env(hq_url)
    hq_client = hq.instance_client.HqInstanceClient(env.hq_url)
    spec = get_instance_spec(hq_client, env)

    if env.use_spec:
        status_cacher = make_status_cacher_from_spec(spec)
        job_ctrl = make_job_ctrl_from_spec(spec, env, status_cacher)
        status_updater = make_status_updater_from_spec(job_ctrl, status_cacher)
    else:
        conf = config.get_or_compile_instance_config(config_path, env)
        status_cacher = confutil.make_status_cacher_from_config(conf)
        job_ctrl = confutil.make_job_ctrl_from_config(conf, spec, env, status_cacher)
        status_updater = confutil.make_status_updater_from_config(conf, job_ctrl, status_cacher, env)
    hq_status_reporter = make_hq_status_reporter(env, status_cacher, hq_client, env.instance_id)
    rotater = make_stdout_rotater(job_ctrl, env)
    return start.StartAction(lock_file=lock,
                             job_ctrl=job_ctrl,
                             status_cacher=status_cacher,
                             status_updater=status_updater,
                             hq_status_reporter=hq_status_reporter,
                             stdout_rotater=rotater)


def make_uninstall_action(config_path, hq_url):
    """
    :type config_path: unicode
    :type hq_url: unicode | types.NoneType
    :rtype: instancectl.actions.install.UninstallAction
    """
    env = envutil.make_instance_ctl_env(hq_url)
    hq_client = hq.instance_client.HqInstanceClient(env.hq_url)
    spec = get_instance_spec(hq_client, env)
    if env.use_spec:
        status_cacher = make_status_cacher_from_spec(spec)
        job_ctrl = make_job_ctrl_from_spec(spec, env, status_cacher)
    else:
        conf = config.get_or_compile_instance_config(config_path, env)
        status_cacher = confutil.make_status_cacher_from_config(conf)
        job_ctrl = confutil.make_job_ctrl_from_config(conf, spec, env, status_cacher)
    return uninstall.UninstallAction(job_ctrl)
