from __future__ import unicode_literals

import collections
import os
import shlex

from yp_proto.yp.client.hq.proto import types_pb2
from sepelib.util import retry

from instancectl.lib import envutil
from instancectl.clients.vault import client as vault
from instancectl.clients.yav import client as yav
from instancectl.coredumps import sender
from instancectl.hq.volumes import its
from instancectl.hq.volumes import template
from instancectl.hq.volumes import secret
from instancectl.jobs import helpers as job_helpers, job, jobpool, controller
from instancectl import errors
from instancectl.lib import netutil
from instancectl.lib import specutil
from instancectl.lib.process.porto_container import PortoMode, VirtMode
from instancectl.status import cacher
from instancectl.status.updater import container
from instancectl.status.updater import container_legacy
from instancectl.status.updater import tcp_check_legacy
from instancectl.status.updater import instance as instance_updater
from instancectl import utils
from instancectl import constants
from instancectl.hq import federated_client
from instancectl.hq import helpers as hq_helpers

from instancectl.sd import client as sd_client
from instancectl.sd import helpers as sd_helpers


def make_container_env(env, spec):
    """
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :type spec: clusterpb.types_pb2.Container
    :rtype: dict[unicode, unicode]
    """
    r = env.default_container_env.copy()
    r['INSTANCECTL_CONTAINER'] = spec.name
    return r


def _is_porto_app_container_enabled(rev, spec):
    if rev.type == types_pb2.InstanceRevision.APP_CONTAINER:
        return True
    if spec.resource_allocation.limit or spec.resource_allocation.request:
        return True
    if spec.coredump_policy.type != types_pb2.CoredumpPolicy.NONE:
        return True
    if spec.security_context.run_as_user:
        return True
    porto_access_policy = spec.security_context.porto_access_policy
    if porto_access_policy and porto_access_policy != 'true':
        return True
    return False


def make_porto_mode_from_spec(rev, spec):
    """
    :type spec: clusterpb.types_pb2.Container
    :type rev: clusterpb.types_pb2.InstanceRevision
    """
    porto_policy = spec.security_context.porto_access_policy
    if rev.type == types_pb2.InstanceRevision.OS_CONTAINER and spec.name == 'os':
        if porto_policy != 'isolate':
            raise errors.ContainerSpecProcessError('OS_CONTAINER must have porto access policy set to "isolate" '
                                                   'instead of current "{}"'.format(porto_policy))
        return PortoMode(enabled=True, isolate='true', virt_mode=VirtMode.OS)
    if _is_porto_app_container_enabled(rev, spec):
        return PortoMode(enabled=True, isolate='false', virt_mode=VirtMode.APP)
    return PortoMode(enabled=False, isolate='false', virt_mode=VirtMode.APP)


def make_job_from_spec(spec, env, rev, sleeper, successful_start_timeout):
    """
    :type spec: clusterpb.types_pb2.Container
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :type rev: clusterpb.types_pb2.InstanceRevision
    :type sleeper: retry.RetrySleeper
    :type successful_start_timeout: int | float | None
    :rtype: instancectl.jobs.job.Job
    """
    env_dict = make_container_env(env, spec)
    work_dir = rev.work_dir or env.instance_dir
    porto_mode = make_porto_mode_from_spec(rev, spec)
    return job.Job(
        spec=spec,
        env=env,
        minidump_sender=None,
        limits={},
        environment=env_dict,
        successful_start_timeout=successful_start_timeout,
        install_script=None,
        restart_script=None,
        uninstall_script=None,
        args_to_eval={},
        rename_binary=None,
        coredump_probability=None,
        always_coredump=False,
        coredumps_dir=constants.DEFAULT_COREDUMPS_DIR,
        expand_spec=True,
        porto_mode=porto_mode,
        work_dir=work_dir,
        coredump_filemask='',
        core_files_limit=0,
        restart_sleeper=sleeper
    )


def need_process_liveness_check(g, c):
    if g['use_true_liveness']:
        return True
    if c['status_check_type'] == 'process_liveness':
        return True
    if c['status_check_type'] == 'list' and 'process_liveness' in c['status_check_list']:
        return True
    return False


def list_optional_checks(g, c):
    """
    :type g: dict
    :type c: dict
    :rtype: list[unicode]
    """
    check_type = c.get('status_check_type')
    if not check_type:
        r = []
    elif check_type != 'list':
        r = [check_type.strip()]
    elif not c['status_check_list']:
        raise errors.ConfigProcessError('status_check_type = list selected but status_check_list is empty')
    else:
        r = [s.strip() for s in c['status_check_list'].split(',')]
    if c['status_script'] and 'script' not in r:
        if g['use_true_liveness']:
            raise errors.ConfigProcessError('status_script is given but status_check_type is not "script"')
        else:
            r.append('script')
    return r


def make_probe_from_config(g, c, env):
    """
    :type g: dict
    :type c: dict
    :rtype: clusterpb.types_pb2.Probe
    """
    p = types_pb2.Probe()
    p.initial_delay_seconds = utils.parse_config_int(c, 'status_check_required_lifetime')
    p.min_period_seconds = int(g['status_update_min_restart_period'])
    p.max_period_seconds = int(g['status_update_max_restart_period'])
    p.period_backoff = int(g['status_update_restart_period_backoff'])
    checks = list_optional_checks(g, c)
    for check in checks:
        if check == 'process_liveness':
            continue
        elif check == 'tcp_check':
            h = p.handlers.add()
            h.type = types_pb2.Handler.TCP_SOCKET
            h.tcp_socket.port = unicode(env.instance_port)
            h.tcp_socket.host = env.hostname
        elif check == 'script':
            h = p.handlers.add()
            h.type = types_pb2.Handler.EXEC
            if not c['status_script']:
                raise errors.ConfigProcessError('Status check "script" set but "status_script" is not given')
            h.exec_action.command.extend(['/bin/sh', '-c', c['status_script']])
        else:
            raise errors.ConfigProcessError('Unknown status check type: {}'.format(check))
    return p


def make_status_updater_from_config(conf, job_ctrl, status_cacher, env):
    """
    :type conf: dict
    :type job_ctrl: instancectl.jobs.controller.JobController
    :type status_cacher: instancectl.status.cacher.InstanceRevisionStatusCacher
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: instancectl.status.updater.instance.InstanceRevisionStatusUpdater
    """
    g = conf['globals']
    updaters = []
    need_legacy_tcp_check = True
    for n, c in conf['jobs'].iteritems():
        probe = make_probe_from_config(g, c, env)
        liveness_check = need_process_liveness_check(g, c)
        if probe.handlers or liveness_check:
            need_legacy_tcp_check = False
        j = job_ctrl.job_pool.jobs[n]
        if liveness_check:
            u = container.ContainerStatusUpdater(job=j, spec=probe, cacher=status_cacher)
        else:
            u = container_legacy.LegacyContainerStatusUpdater(job=j, spec=probe, cacher=status_cacher)
        updaters.append(u)
    if need_legacy_tcp_check:
        sleeper = retry.RetrySleeper(delay=g['status_update_min_restart_period'],
                                     backoff=g['status_update_restart_period_backoff'],
                                     max_delay=g['status_update_max_restart_period'],
                                     max_jitter=0)
        spec = types_pb2.TCPSocketAction()
        spec.port = unicode(env.instance_port)
        spec.host = env.hostname
        u = tcp_check_legacy.LegacyTcpActionStatusUpdater(spec, status_cacher, sleeper, job_ctrl)
        updaters.append(u)
    return instance_updater.InstanceRevisionStatusUpdater(updaters)


def make_status_cacher_from_config(conf):
    """
    :type conf: dict
    :rtype: instancectl.status.cacher.InstanceRevisionStatusCacher
    """
    g = conf['globals']
    if g['use_true_liveness']:
        criterion = g['liveness_criterion_type']
        if criterion == 'list':
            if not g['liveness_criterion_list']:
                raise errors.ConfigProcessError('"liveness_criterion_type = list" but liveness_criterion_list is empty')
            containers = g['liveness_criterion_list'].split(',')
            for c in containers:
                c = c.strip()
                if c not in conf['jobs']:
                    raise errors.ConfigProcessError('Unknown section name "{}" in liveness_criterion_list'.format(c))
        elif criterion == 'all':
            containers = list(conf['jobs'])
        else:
            raise errors.ConfigProcessError('Unknown liveness criterion type: "{}"'.format(criterion))
    else:
        containers = []
        for n, c in conf['jobs'].iteritems():
            checks = list_optional_checks(g, c)
            if checks:
                containers.append(n)
    s = types_pb2.RevisionStatus()
    s.installed.status = constants.CONDITION_FALSE
    s.installed.reason = 'RevisionNotInstalled'
    s.installed.last_transition_time.GetCurrentTime()
    s.ready.status = constants.CONDITION_FALSE
    s.ready.reason = 'RevisionNotReady'
    s.ready.last_transition_time.GetCurrentTime()
    for j in conf['jobs']:
        cont = s.container.add()
        cont.name = j
        cont.ready.status = constants.CONDITION_FALSE
        cont.ready.reason = 'RevisionNotReady'
        cont.ready.last_transition_time.GetCurrentTime()
        cont.installed.status = constants.CONDITION_FALSE
        cont.installed.reason = 'RevisionNotInstalled'
        cont.installed.last_transition_time.GetCurrentTime()
    return cacher.InstanceRevisionStatusCacher(status=s, criterion_containers=containers)


def make_minidump_sender_from_config(section, conf, limits):
    """
    :type section: unicode
    :type conf: dict
    :type limits: dict
    :rtype: instancectl.coredumps.sender.MinidumpSender
    """
    aggr_type = sender.AggregatorType.__members__[conf['minidumps_aggregator'].upper()]
    core_type = sender.Backend.__members__[conf['coredumps_format'].upper()]
    return sender.MinidumpSender(
        section=section,
        run_config=conf,
        aggregator_type=aggr_type,
        limits=limits,
        coredump_format=core_type,
    )


def make_job_ctrl_from_config(conf, spec, env, status_cacher):
    """
    :type conf: dict
    :type spec: clusterpb.types_pb2.InstanceRevision
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: instancectl.jobs.controller.JobController
    """
    g = conf['globals']

    federated_url = g['federated_url']
    vault_url = g['vault_url']
    yav_url = g['yav_url']
    tvm_api_url = g['tvm_api_url']
    tvm_client_id = g['tvm_client_id']
    yav_tvm_client_id = g['yav_tvm_client_id']

    vault_client = vault.VaultClient(vault_url, service_id=env.service_id)
    yav_client = yav.YavClient(yav_url,
                               service_id=env.service_id,
                               tvm_client_id=tvm_client_id,
                               yav_tvm_client_id=yav_tvm_client_id,
                               tvm_api_url=tvm_api_url)

    # Instance volume plugins
    context = make_template_plugin_context(federated_url, env)
    template_volume_plugin = template.TemplateVolumePlugin(context, instance_dir='.')
    secret_volume_plugin = secret.SecretVolumePlugin()

    controls_real_path = g['its_symlink_controls']
    if controls_real_path:
        controls_real_path = os.path.join(controls_real_path, env.instance_id)
        its_shared_storage = None
    else:
        its_shared_storage = g['its_shared_storage'] or envutil.make_its_shared_storage_dir(env)
    its_volume_plugin = its.ItsVolumePlugin(service_id=env.service_id,
                                            auto_tags=env.auto_tags,
                                            first_poll_timeout=constants.ITS_FIRST_POLL_TIMEOUT,
                                            shared_storage=its_shared_storage,
                                            controls_real_path=controls_real_path)

    if g['its_poll']:
        v = spec.volume.add()
        v.name = constants.ITS_CONTROLS_DIR
        v.type = types_pb2.Volume.ITS
        v.its_volume.its_url = g['its_url']
        v.its_volume.period_seconds = int(g['its_force_poll_timeout'])
        v.its_volume.max_retry_period_seconds = int(g['its_max_poll_timeout'])

    jobs = collections.OrderedDict()
    init_containers = []
    container_specs = {c.name: c for c in spec.container}
    # Run spec containers before prepare_script
    for c in spec.init_containers:
        sleeper = envutil.create_restart_sleeper(env.prepare_script_restart_policy)
        cont = make_job_from_spec(c, env, spec, sleeper, successful_start_timeout=None)
        init_containers.append(cont)
    for section, s in conf['jobs'].iteritems():
        limits = job_helpers.make_limits_dict_from_config(s)
        minidump_sender = make_minidump_sender_from_config(section, s, limits)
        args_to_eval = {}
        for k, v in s.iteritems():
            if k.startswith('eval_'):
                name = k[len('eval_'):]
                args_to_eval[name] = v
        c_spec = container_specs.get(section, types_pb2.Container(name=section))
        # I don't know how to do it better
        c_spec.command[:] = [s['binary']] + shlex.split(s['arguments']) + s['extra_run_arguments']
        c_spec.restart_policy.min_period_seconds = int(s['delay'])
        c_spec.restart_policy.max_period_seconds = int(s['max_delay'])
        c_spec.restart_policy.period_backoff = int(s['backoff'])
        c_spec.restart_policy.period_jitter_seconds = int(s['max_jitter'])
        c_spec.lifecycle.stop_grace_period_seconds = int(s['terminate_timeout'])
        c_spec.lifecycle.termination_grace_period_seconds = int(s['kill_timeout'])
        stop_script = s['stop_script']
        if stop_script:
            c_spec.lifecycle.pre_stop.type = types_pb2.Handler.EXEC
            c_spec.lifecycle.pre_stop.exec_action.command.extend(['/bin/sh', '-c', stop_script])
        reopenlog_script = s['reopenlog_script']
        if reopenlog_script:
            h = c_spec.reopen_log_action.handler
            h.type = types_pb2.Handler.EXEC
            h.exec_action.command.extend(['/bin/sh', '-c', reopenlog_script])
        if not s['use_porto'].enabled:
            s['use_porto'] = make_porto_mode_from_spec(spec, c_spec)
        if s['use_porto'].enabled:
            if s['use_porto'].virt_mode == VirtMode.OS:
                c_spec.security_context.run_as_user = 'root'
                c_spec.security_context.porto_access_policy = 'isolate'
            else:
                c_spec.security_context.porto_access_policy = 'true'
        specutil.prepare_security_context(c_spec.security_context, env)
        sleeper = specutil.create_restart_sleeper(c_spec)
        j = job.Job(
            spec=c_spec,
            env=env,
            minidump_sender=minidump_sender,
            limits=limits,
            environment=s['environment'].copy(),
            successful_start_timeout=float(s['successful_start_timeout']),
            install_script=s['install_script'],
            restart_script=s['restart_script'],
            uninstall_script=s['uninstall_script'],
            args_to_eval=args_to_eval,
            rename_binary=s['rename_binary'],
            coredump_probability=s['coredump_probability'],
            always_coredump=s['always_coredump'],
            coredumps_dir=s['coredumps_dir'],
            expand_spec=False,
            porto_mode=s['use_porto'],
            work_dir=env.instance_dir,
            coredump_filemask=s['coredumps_filemask'],
            core_files_limit=s['coredumps_count_limit'],
            restart_sleeper=sleeper
        )
        jobs[section] = j
        if s['prepare_script']:
            init_spec = types_pb2.Container()
            init_spec.name = 'prepare_{}'.format(section)
            init_spec.command.extend(['/bin/sh', '-c', s['prepare_script']])
            init_spec.restart_policy.type = types_pb2.RestartPolicy.ON_FAILURE
            prepare_sleeper = envutil.create_restart_sleeper(env.prepare_script_restart_policy)
            j = job.Job(
                spec=init_spec,
                env=env,
                minidump_sender=sender.EmptyMinidumpSender(),
                limits=limits,
                environment=s['environment'].copy(),
                successful_start_timeout=None,
                install_script=None,
                restart_script=None,
                uninstall_script=None,
                args_to_eval={},
                rename_binary=None,
                coredump_probability=0,
                always_coredump=False,
                coredumps_dir=constants.DEFAULT_COREDUMPS_DIR,
                expand_spec=False,
                porto_mode=PortoMode(enabled=False, isolate='false', virt_mode=VirtMode.APP),
                work_dir=env.instance_dir,
                coredump_filemask='',
                core_files_limit=0,
                restart_sleeper=prepare_sleeper
            )
            init_containers.append(j)
        if s['notify_script']:
            h = spec.notify_action.handlers.add()
            h.type = types_pb2.Handler.EXEC
            h.exec_action.command.extend(['/bin/sh', '-c', s['notify_script']])
    for c, s in container_specs.iteritems():
        if c in constants.AUX_DAEMONS_NAMES:
            if c in jobs:
                raise errors.ContainerSpecProcessError('Container "{}" is given in instancectl config and selected in '
                                                       'Instance Spec.'.format(c))
            sleeper = specutil.create_restart_sleeper(s)
            jobs[c] = make_job_from_spec(
                s, env, spec, sleeper, constants.DEFAULT_CONTAINER_RESTART_SUCCESSFUL_START_TIMEOUT
            )
            continue
        if c not in conf['jobs']:
            raise errors.ContainerSpecProcessError('Container "{}" is given in service Instance Spec but there is no '
                                                   'such section in instancectl config. Please remove it from Instance '
                                                   'Spec or add to instancectl config.'.format(c))
    job_pool = jobpool.JobPool(
        jobs=jobs,
        spec=spec,
        init_containers=init_containers,
        secret_volume_plugin=secret_volume_plugin,
        template_volume_plugin=template_volume_plugin,
        vault_client=vault_client,
        yav_client=yav_client,
        status_cacher=status_cacher,
        its_volume_plugin=its_volume_plugin,
        env=env,
    )
    return controller.JobController(job_pool=job_pool,
                                    action_stop_timeout=g['action_stop_timeout'])


def make_template_plugin_context(federated_url, env):
    """
    :type federated_url: unicode
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: dict
    """
    federated = federated_client.FederatedClient(url=federated_url)
    hq_getter = hq_helpers.HqInstancesGetter(federated_client=federated, env=env)
    sd_getter = sd_helpers.SdInstancesGetter(
        sd_client=sd_client.SdGrpcClient(env.instance_id, sd_url=env.sd_url),
        env=env
    )
    context = env.default_container_env.copy()
    context['get_hq_instances'] = hq_getter.get_instances
    context['get_hq_instances_current_cluster'] = hq_getter.get_instances_current_cluster

    context['get_sd_endpoints'] = sd_getter.get_endpoints
    context['get_sd_endpoints_current_cluster'] = sd_getter.get_endpoints_current_cluster

    context['get_sd_pods'] = sd_getter.get_pods
    context['get_sd_pods_current_cluster'] = sd_getter.get_pods_current_cluster

    context['orthogonal_tags'] = env.orthogonal_tags
    try:
        context['CURRENT_REV'] = envutil.get_configuration_id()
    except envutil.InstanceCtlEnvironmentError:
        context['CURRENT_REV'] = ''
    context['resolve_ipv6_addr'] = netutil.resolve_ipv6_addr
    context['resolve_ipv4_addr'] = netutil.resolve_ipv4_addr
    return context
