# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import base64
import logging
import os
import re
import shlex
import socket

import enum
import requests

from infra.nanny.instancectl.proto import cluster_api_pb2

from sepelib.util import retry

from instancectl import constants
from instancectl import cms
from instancectl.clients.unixsocket_rpc.adapters import UnixAdapter, UnixHttp1Adapter
from instancectl.clients.unixsocket_rpc import client


log = logging.getLogger('envutil')


GEO_SAS_HOSTNAME_REGEX = re.compile(r'sas\d-\d+.search.yandex.net')
GEO_MAN_HOSTNAME_REGEX = re.compile(r'man\d-\d+.search.yandex.net')
GEO_VLA_HOSTNAME_REGEX = re.compile(r'vla\d-\d+.search.yandex.net')


class InstanceCtlEnvironmentError(Exception):
    pass


class Location(enum.Enum):
    UNKNOWN = 0
    MSK = 1
    SAS = 2
    MAN = 3
    VLA = 4


LOCATION_TO_HQ_DEFAULT_URL_OPTION_MAP = {
    Location.MSK: 'http://hq.msk-swat.yandex-team.ru/',
    Location.SAS: 'http://hq.sas-swat.yandex-team.ru/',
    Location.MAN: 'http://hq.man-swat.yandex-team.ru/',
    Location.VLA: 'http://hq.vla-swat.yandex-team.ru/',
    Location.UNKNOWN: 'http://hq.msk-swat.yandex-team.ru/',
}

LOCATION_TO_HQ_SWAT_7267_URL_OPTION_MAP = {
    Location.MSK: 'http://hq-fallback-msk.nanny.yandex-team.ru/',
    Location.SAS: 'http://hq-fallback-sas.nanny.yandex-team.ru/',
    Location.MAN: 'http://hq-fallback-man.nanny.yandex-team.ru/',
    Location.VLA: 'http://hq-fallback-vla.nanny.yandex-team.ru/',
    Location.UNKNOWN: 'http://hq-fallback-msk.nanny.yandex-team.ru/',
}

YSON_STRING_TOKEN = b'\x01'


def _parse_yson_string(s):
    # Copy-pasted from yt.yson
    count = 1  # Strip \x01
    read_next = True
    while read_next:
        ch = s[count]
        if not ch:
            raise ValueError('Cannot parse yson string')
        byte = ord(ch)
        count += 1
        read_next = byte & 0x80 != 0
    return s[count:]


def extract_location_from_hostname(hostname):
    """
    :rtype: Location
    """
    if GEO_SAS_HOSTNAME_REGEX.match(hostname):
        return Location.SAS
    elif GEO_MAN_HOSTNAME_REGEX.match(hostname):
        return Location.MAN
    elif GEO_VLA_HOSTNAME_REGEX.match(hostname):
        return Location.VLA
    return Location.UNKNOWN


def extract_location_from_orthogonal_tags(tags):
    """
    :type tags: dict[str | unicode, str | unicode]
    :rtype: Location
    """
    tag = tags.get('a_geo', '').upper()
    return Location.__members__.get(tag, Location.UNKNOWN)


def extract_dc_from_orthogonal_tags(tags):
    """
    :type tags: dict[str | unicode, str | unicode]
    :rtype: unicode
    """
    tag = tags.get('a_dc', '').upper()
    return tag


def make_hq_url(orthogonal_tags, hostname):
    """
    :type orthogonal_tags: dict[unicode, unicode]
    :type hostname: unicode
    :rtype: unicode
    """
    location = extract_location_from_orthogonal_tags(orthogonal_tags)
    if location == Location.UNKNOWN:
        location = extract_location_from_hostname(hostname)

    if orthogonal_tags.get('a_itype') == 'balancer' and orthogonal_tags.get('a_prj') == 'swat':
        # Start with https://st.yandex-team.ru/SWAT-7267 for details
        url_map = LOCATION_TO_HQ_SWAT_7267_URL_OPTION_MAP
    else:
        url_map = LOCATION_TO_HQ_DEFAULT_URL_OPTION_MAP

    url = url_map.get(location)
    if url:
        return url
    return url_map[Location.UNKNOWN]


def _make_service_id(conf_id):
    """
    :type conf_id: unicode | None
    :rtype: unicode
    """
    s_id = os.environ.get(constants.NANNY_SERVICE_ID_OPTION)
    if s_id:
        return s_id

    if conf_id:
        return conf_id.rsplit('-', 1)[0]


def get_configuration_id():
    """
    Returns current configuration name. Works under ISS and BSCONFIG.
    Under ISS it is part after '#' sign, e.g. it's "conf-123" if instance configuration id is "conf#conf-123".

    Attention! Because of dynamic resources, configuration name may be changed on the fly!

    :rtype: unicode
    :raises: InstanceCtlEnvironmentError
    """
    if cms.detect_cms_agent_type().is_iss():
        dump_json = cms.load_dump_json()
        conf_id = dump_json.get('configurationId')
        if conf_id is None:
            raise InstanceCtlEnvironmentError('Cannot parse out configurationId from ISS dump.json')
        return conf_id.rsplit('#', 1)[-1]
    bsconfig_dir = os.environ.get('BSCONFIG_IDIR')
    if bsconfig_dir is None:
        raise InstanceCtlEnvironmentError('Cannot parse out configuration from BSCONFIG_IDIR environment variable')
    result = re.compile('/bsconfig/configinstall/([-\w]+)/').search(bsconfig_dir)
    if not result:
        raise InstanceCtlEnvironmentError('Cannot parse out configuration from BSCONFIG_IDIR environment variable')
    return result.groups()[0]


def get_configuration_id_ignore():
    """
    :rtype: unicode | None
    """
    try:
        return get_configuration_id()
    except InstanceCtlEnvironmentError:
        return None


def get_orthogonal_tags_dict(auto_tags):
    """
    Составляет словарь ортогональных тегов, получаемых из dump.json:
    {
        "geo": "msk",
        "ctype": "prod",
        ...
    }

    Функция обрезает множественный a_prj_

    :type auto_tags: list[str | unicode]
    :rtype: dict[str | unicode, str | unicode]
    """
    # tags example:
    #  "tags" : "MSK_SG_CLUSTER_ALEMATE_WORKER a_ctype_prod a_dc_fol a_geo_msk a_itype_alematedworker
    #  a_line_fol-7 a_metaprj_internal a_prj_alemate a_tier_none
    #  a_topology_group-MSK_SG_CLUSTER_ALEMATE_WORKER
    #  a_topology_trunk-2270085 a_topology_version-trunk-2270085"

    result = {}
    for known_tag in constants.ORTHOGONAL_TAG_LIST:
        for tag in auto_tags:
            if not tag.startswith(known_tag):
                continue
            result[known_tag] = tag[len(known_tag) + 1:]
            break
        if known_tag not in result:
            result[known_tag] = 'undefined'

    return result


def get_auto_tags(tags_string):
    """
    :type tags_string: unicode
    :rtype: list[unicode]
    """
    return sorted(t for t in shlex.split(tags_string) if t.startswith('a_'))


def make_gencfg_ports_render_vars(main_port):
    """
    :type main_port: int
    :rtype: dict[unicode, int]
    """
    return {'BSCONFIG_IPORT_PLUS_{}'.format(i): unicode(main_port + i) for i in xrange(1, 21)}


def make_container_env(port, instance_dir, tags, instance_name, bsconfig_ihost, orthogonal_tags, node_name,
                       deploy_engine, yp_pod_spec=None):
    """
    :type port: int
    :type instance_dir: unicode
    :type tags: unicode
    :type instance_name: unicode
    :type orthogonal_tags: dict[unicode, unicode]
    :type bsconfig_ihost: unicode
    :type node_name: unicode
    :type deploy_engine: unicode
    :type yp_pod_spec: dict | None
    :rtype: dict
    """
    port_vars = make_gencfg_ports_render_vars(port)
    env = os.environ.copy()
    env.update({
        'BSCONFIG_IDIR': instance_dir,
        'BSCONFIG_IPORT': unicode(port),
        'BSCONFIG_SHARDNAME': os.environ.get('BSCONFIG_SHARDNAME', ''),
        'BSCONFIG_SHARDDIR': os.environ.get('BSCONFIG_SHARDDIR', './'),
        'BSCONFIG_INAME': instance_name,
        'BSCONFIG_ITAGS': tags,
        'BSCONFIG_IHOST': bsconfig_ihost,
        'NODE_NAME': node_name,
    })
    env.update(orthogonal_tags)
    env.update(port_vars)
    if deploy_engine == 'YP_LITE':
        try:
            yp_lite_envs = _make_yp_lite_env_vars(yp_pod_spec)
        except Exception as e:
            log.exception('Cannot read YP-lite specific settings from ISS-agent unixsocket')
            raise InstanceCtlEnvironmentError(
                'Cannot read YP-lite specific settings from ISS-agent unixsocket: error: "{}"'.format(e))
        env.update(yp_lite_envs)
    return env


def make_its_shared_storage_dir(env):
    """
    :type env: InstanceCtlEnv
    """
    cms_type = cms.detect_cms_agent_type()

    if cms_type.cms == cms.CmsType.BSCONFIG:
        cms_path = constants.ITS_BSCONFIG_SHARED_STORAGE_PATH
    elif cms_type.cms == cms.CmsType.ISS:
        cms_path = constants.ITS_ISS_SHARED_STORAGE_PATH
    else:
        raise ValueError('Unknown CMS type, must be bsconfig or iss')

    orthogonal_tags_dict = env.orthogonal_tags
    joined_orthogonal_tags = '-'.join(orthogonal_tags_dict[x] for x in constants.ITS_STATE_VOLUME_NAME_TAG_LIST)

    return constants.ITS_SHARED_STORAGE_TEMPLATE.format(
        cms_path=cms_path,
        host=env.hostname,
        port=env.instance_port,
        tags=joined_orthogonal_tags,
    )


def _make_instance_port(deploy_engine):
    """
    :rtype: int
    """
    env_port = os.environ.get('BSCONFIG_IPORT')
    if env_port is not None:
        return int(env_port)
    if deploy_engine == 'YP_LITE':
        return constants.YP_LITE_DEFAULT_INSTANCE_PORT
    raise InstanceCtlEnvironmentError('Internal Nanny error: cant determine instance port and deploy engine is not YP')


def _get_pod_spec_from_unixsocket(unix_adapter_cls=UnixAdapter):
    """
    :type unix_adapter_cls: type
    :rtype: dict
    """
    s = requests.Session()
    log.info('Reading pod spec from ISS-agent unixsocket')
    s.mount(client.UNIX_SOCKET_SCHEME, unix_adapter_cls())
    sleeper = retry.RetrySleeper(delay=1.0, backoff=2, max_delay=5, max_tries=4)
    retry_with_timeout = retry.RetryWithTimeout(attempt_timeout=5.0, retry_sleeper=sleeper)
    url = '{}{}/pod_spec'.format(client.UNIX_SOCKET_SCHEME, constants.NODE_AGENT_YP_SOCKET_PATH)
    resp = retry_with_timeout(s.get, url)
    log.info('Successfully read pod spec from ISS-agent unixsocket')
    return resp.json()


def _parse_labels_from_pod_spec(pod_spec):
    dyn_attrs = pod_spec.get('podDynamicAttributes')
    if not dyn_attrs:
        return {}
    labels = dyn_attrs.get('labels')
    if not labels:
        return {}
    attrs = labels.get('attributes', [])
    rv = {}
    for a in attrs:
        key = a['key']
        env_var_name = 'LABELS_{}'.format(key)
        # Protobuf encodes bytes as base64 so we have to parse it
        b64_encoded = a['value']
        yson_encoded = base64.b64decode(b64_encoded)
        # https://wiki.yandex-team.ru/yt/userdoc/yson/ YSON strings are encoded as:
        # \x01 + length (protobuf sint32 wire format) + data (<length> bytes)
        # So we have to check if the first byte is \x01 and ignore label otherwise
        if yson_encoded.startswith(YSON_STRING_TOKEN):
            parsed = _parse_yson_string(yson_encoded)
            rv[env_var_name] = parsed
    return rv


def _make_yp_lite_env_vars(data):
    backbone_ip_address = None
    for allocation in data['ip6AddressAllocations']:
        if allocation['vlanId'] == 'backbone':
            backbone_ip_address = allocation['address']
            break
    if not backbone_ip_address:
        raise InstanceCtlEnvironmentError('Backbone IP address is not given in pod spec payload')
    rv = {'BACKBONE_IP_ADDRESS': backbone_ip_address}
    for p in data['portoProperties']:
        k = p['key']
        v = p['value']
        if k == 'cpu_limit':
            rv['CPU_LIMIT'] = v
        elif k == 'cpu_guarantee':
            rv['CPU_GUARANTEE'] = v
        elif k == 'memory_limit':
            rv['MEM_LIMIT'] = v
        elif k == 'memory_guarantee':
            rv['MEM_GUARANTEE'] = v
    labels = _parse_labels_from_pod_spec(data)
    rv.update(labels)
    return rv


def create_restart_sleeper(restart_policy):
    delay = restart_policy.get("delay")
    max_delay = restart_policy.get("max_delay")
    backoff = restart_policy.get("backoff")
    max_jitter = restart_policy.get("max_jitter")
    max_tries = restart_policy.get("max_tries")
    return retry.RetrySleeper(
        delay=delay,
        max_delay=max_delay,
        backoff=backoff,
        max_jitter=max_jitter,
        max_tries=max_tries
    )


def get_yp_hq_spec(yp_pod_spec, conf_id, box_id):
    """
    :type yp_pod_spec: dict
    :type conf_id: unicode
    :type box_id: unicode | NoneType
    :rtype: yp_proto.yp.client.hq.proto.types_pb2.InstanceRevision | None
    """
    iss_payload_b64encoded = yp_pod_spec.get('issPayload', None)
    if iss_payload_b64encoded is None:
        return None

    iss_payload_proto = cluster_api_pb2.HostConfiguration()
    iss_payload_proto.ParseFromString(base64.b64decode(iss_payload_b64encoded))

    for instance in iss_payload_proto.instances:
        if box_id is not None:
            instance_box_id = instance.properties.get('BOX_ID')
            if instance_box_id != box_id:
                continue
        if conf_id == instance.id.configuration.groupStateFingerprint and instance.HasField('instanceRevision'):
            return instance.instanceRevision


def get_from_env(key, value_type, default=None):
    value = os.environ.get(key)
    if value is None:
        return default
    try:
        return value_type(value)
    except ValueError:
        return default


def make_instance_ctl_env(hq_url):
    """
    :type hq_url: unicode | types.NoneType
    :rtype: InstanceCtlEnv
    """
    hostname = os.environ.get('HOSTNAME') or socket.gethostname()
    node_name = os.environ.get('YP_NODE_FQDN') or os.environ.get('NODE_NAME') or hostname
    deploy_engine = os.environ.get('DEPLOY_ENGINE')
    instance_port = _make_instance_port(deploy_engine)
    conf_id_or_none = get_configuration_id_ignore()
    service_id = _make_service_id(conf_id_or_none)
    hq_id = os.environ.get('HQ_INSTANCE_ID')
    instance_id = hq_id or '{node}:{port}@{service}'.format(node=node_name, port=instance_port, service=service_id)
    tags_string = os.environ.get('BSCONFIG_ITAGS') or os.environ.get('tags', '')
    auto_tags = get_auto_tags(tags_string)
    orthogonal_tags = get_orthogonal_tags_dict(auto_tags)
    use_spec = constants.RUN_CONTAINERS_FROM_HQ_SPEC in tags_string
    hq_report = constants.ENABLE_HQ_REPORT in tags_string
    hq_poll = constants.ENABLE_HQ_POLL in tags_string
    instance_dir = os.environ.get('BSCONFIG_IDIR') or os.getcwd()
    instance_name = os.environ.get('BSCONFIG_INAME') or instance_id.split('@', 1)[0]
    bsconfig_ihost = os.environ.get('BSCONFIG_IHOST') or node_name.split('.', 1)[0]
    skip_fdatasync_config = bool(os.environ.get('SKIP_FDATASYNC_CONFIG'))
    box_id = os.environ.get('BOX_ID')
    mock_retry_sleeper_output = os.environ.get('INSTANCECTL_MOCK_RETRY_SLEEPER_OUTPUT')

    prepare_script_restart_policy = {
        "backoff": get_from_env('PREPARE_SCRIPT_BACKOFF', float, constants.DEFAULT_PREPARE_SCRIPT_BACKOFF),
        "max_tries": get_from_env('PREPARE_SCRIPT_MAX_TRIES', int, constants.DEFAULT_PREPARE_SCRIPT_MAX_TRIES),
        "delay": get_from_env('PREPARE_SCRIPT_DELAY', float, constants.DEFAULT_PREPARE_SCRIPT_MIN_DELAY),
        "max_delay": get_from_env('PREPARE_SCRIPT_MAX_DELAY', float, constants.DEFAULT_PREPARE_SCRIPT_MAX_DELAY),
        "max_jitter": get_from_env('PREPARE_SCRIPT_MAX_JITTER', float, constants.DEFAULT_PREPARE_SCRIPT_MAX_JITTER)
    }
    install_script_restart_policy = {
        "backoff": get_from_env('INSTALL_SCRIPT_BACKOFF', float, constants.DEFAULT_INSTALL_SCRIPT_BACKOFF),
        "max_tries": get_from_env('INSTALL_SCRIPT_MAX_TRIES', int, constants.DEFAULT_INSTALL_SCRIPT_MAX_TRIES),
        "delay": get_from_env('INSTALL_SCRIPT_DELAY', float, constants.DEFAULT_INSTALL_SCRIPT_MIN_DELAY),
        "max_delay": get_from_env('INSTALL_SCRIPT_MAX_DELAY', float, constants.DEFAULT_INSTALL_SCRIPT_MAX_DELAY),
        "max_jitter": get_from_env('INSTALL_SCRIPT_MAX_JITTER', float, constants.DEFAULT_INSTALL_SCRIPT_MAX_JITTER)
    }

    yp_pod_spec = None
    if deploy_engine == 'YP_LITE':
        # UnixHttp1Adapter is hot fix for https://st.yandex-team.ru/ISS-7152
        yp_pod_spec = _get_pod_spec_from_unixsocket(unix_adapter_cls=UnixHttp1Adapter)

    default_env = make_container_env(port=instance_port,
                                     instance_dir=instance_dir,
                                     tags=tags_string,
                                     instance_name=instance_name,
                                     bsconfig_ihost=bsconfig_ihost,
                                     orthogonal_tags=orthogonal_tags,
                                     node_name=node_name,
                                     deploy_engine=deploy_engine,
                                     yp_pod_spec=yp_pod_spec)
    hq_url = hq_url or make_hq_url(orthogonal_tags, hostname)

    yp_hq_spec = None
    if conf_id_or_none and yp_pod_spec:
        yp_hq_spec = get_yp_hq_spec(yp_pod_spec, conf_id_or_none, box_id)
        log.info("Load hq spec from YP for conf %s: %s", conf_id_or_none,
                 'done' if yp_hq_spec is not None else 'not found')

    hq_report_version = int(os.environ.get('HQ_REPORT_VERSION', '1'))

    sd_url = os.environ.get('SD_URL', None)

    return InstanceCtlEnv(
        instance_port=instance_port,
        instance_name=instance_name,
        instance_dir=instance_dir,
        instance_id=instance_id,
        service_id=service_id,
        node_name=node_name,
        hostname=hostname,
        orthogonal_tags=orthogonal_tags,
        use_spec=use_spec,
        hq_poll=hq_poll,
        hq_report=hq_report,
        default_container_env=default_env,
        hq_url=hq_url,
        auto_tags=auto_tags,
        instance_tags_string=tags_string,
        yp_hq_spec=yp_hq_spec,
        skip_fdatasync_config=skip_fdatasync_config,
        hq_report_version=hq_report_version,
        sd_url=sd_url,
        prepare_script_restart_policy=prepare_script_restart_policy,
        install_script_restart_policy=install_script_restart_policy,
        mock_retry_sleeper_output=mock_retry_sleeper_output
    )


class InstanceCtlEnv(object):

    def __init__(self, instance_port, instance_name, instance_dir, instance_id, service_id, node_name, hostname,
                 orthogonal_tags, use_spec, hq_poll, hq_report, default_container_env, hq_url, auto_tags,
                 instance_tags_string, yp_hq_spec, skip_fdatasync_config, hq_report_version, sd_url,
                 mock_retry_sleeper_output, prepare_script_restart_policy, install_script_restart_policy):
        """

        :type instance_port: int
        :type instance_name: unicode
        :type instance_dir: unicode
        :type instance_id: unicode
        :type service_id: unicode
        :type node_name: unicode
        :type hostname: unicode
        :type orthogonal_tags: dict[unicode, unicode]
        :type use_spec: bool
        :type hq_poll: bool
        :type hq_report: bool
        :type default_container_env: dict[unicode, unicode]
        :type auto_tags: list[unicode]
        :type instance_tags_string: unicode
        :type yp_hq_spec: yp_proto.yp.client.hq.proto.types_pb2.InstanceRevision | None
        :type skip_fdatasync_config: bool
        :type hq_report_version: int
        :type sd_url: unicode
        :type prepare_script_restart_policy: dict
        :type install_script_restart_policy: dict
        """
        self.instance_port = instance_port
        self.instance_name = instance_name
        self.instance_dir = instance_dir
        self.instance_id = instance_id
        self.service_id = service_id
        self.node_name = node_name
        self.hostname = hostname
        self.auto_tags = auto_tags
        self.orthogonal_tags = orthogonal_tags
        self.use_spec = use_spec
        self.hq_poll = hq_poll
        self.hq_report = hq_report
        self.default_container_env = default_container_env
        self.hq_url = hq_url
        self.instance_tags_string = instance_tags_string
        self.yp_hq_spec = yp_hq_spec
        self.skip_fdatasync_config = skip_fdatasync_config
        self.hq_report_version = hq_report_version
        self.sd_url = sd_url
        self.prepare_script_restart_policy = prepare_script_restart_policy
        self.install_script_restart_policy = install_script_restart_policy
        self.mock_retry_sleeper_output = mock_retry_sleeper_output
