from __future__ import unicode_literals

import time

import gevent

from sepelib.subprocess import util

from instancectl.lib import procutil
from . import errors


class ExecutionContext(object):

    def __init__(self, timeout):
        """
        :type timeout: float | types.NoneType
        """
        if timeout is None:
            self.deadline = None
        else:
            self.deadline = time.time() + timeout

    def is_timed_out(self):
        """
        :rtype: bool
        """
        if self.deadline is None:
            return False
        return time.time() > self.deadline


class ExecRunner(object):

    PROCESS_CHECK_TIMEOUT = 0.1

    def __init__(self, stdout, stderr, env, limits, pass_fds, log):
        """
        :type stdout: file
        :type stderr: file
        :type env: dict[unicode, unicode]
        :type limits: dict[unicode, instancectl.common.RLimit]
        :type pass_fds: list[int]
        """
        self.stdout = stdout
        self.stderr = stderr
        self.env = env
        self.preexec_fn = procutil.ExecuteHelper(limits)
        self.pass_fds = pass_fds
        self.log = log

    def run(self, command, timeout, additional_env):
        """
        :param command: script body
        :type command: list[unicode]
        :type timeout: int | types.NoneType
        :type additional_env: dict[unicode, unicode]
        :rtype: int
        """
        env = self.env.copy()
        env.update(additional_env)
        self.log.info('Starting cmd "%s"', command)
        try:
            process = procutil.ExtendedPopen(
                command,
                pass_fds=self.pass_fds,
                close_fds=bool(self.pass_fds),
                stdout=self.stdout,
                stderr=self.stderr,
                preexec_fn=self.preexec_fn,
                shell=False,
                env=env,
            )
        except Exception as e:
            raise errors.ExecRunnerError('Cannot start command: {}'.format(e))
        ctx = ExecutionContext(timeout)
        rc = process.poll()
        while rc is None and not ctx.is_timed_out():
            gevent.sleep(self.PROCESS_CHECK_TIMEOUT)
            rc = process.poll()

        if rc is None:
            try:
                util.terminate(process)
            except Exception as e:
                raise errors.ExecRunnerError('Cannot kill timed out command ({} seconds): {}'.format(timeout, e))
            else:
                raise errors.ExecRunnerTimeoutError('Command timed out ({} seconds) and killed'.format(timeout))

        self.log.info('Cmd exit code: %s', rc)
        if rc != 0:
            raise errors.ExecRunnerExitStatusError('Command failed with status: {}'.format(rc))
