from __future__ import unicode_literals

import google.protobuf.descriptor


FD = google.protobuf.descriptor.FieldDescriptor


_INT_TYPES = (
    FD.TYPE_INT32,
    FD.TYPE_SINT32,
    FD.TYPE_FIXED32,
    FD.TYPE_SFIXED32,
)
_FLOAT_TYPES = (
    FD.TYPE_FLOAT,
    FD.TYPE_DOUBLE
)
# According to https://developers.google.com/protocol-buffers/docs/proto3?hl=en#json
# 64bit types are mapped to strings for some reason.
_STRING_TYPES = (
    FD.TYPE_INT64,
    FD.TYPE_SINT64,
    FD.TYPE_STRING,
    FD.TYPE_FIXED64,
    FD.TYPE_SFIXED64,
)
_BOOL_TYPES = (
    FD.TYPE_BOOL,
)
_STRUCT_TYPES = (
    FD.TYPE_MESSAGE,
)

LABEL_REPEATED = FD.LABEL_REPEATED


def _infer_field_schema(field_desc):
    validation_options = {}
    # Extract message options
    for option, value in field_desc.GetOptions().ListFields():
        validation_options[option.name] = value

    # Copy field options to separate dict to don't overwrite pb 'required' option
    # by jsonschema's required fields list
    r = validation_options.copy()
    r.pop('required', None)

    # Extract default value if present
    # SWAT-4051
    # if field_desc.default_value is not None:
    #    r['default'] = field_desc.default_value
    # Extract field options
    if field_desc.type in _INT_TYPES:
        if field_desc.label == FD.LABEL_REPEATED:
            r['type'] = 'array'
            r['items'] = {'type': 'integer'}
        else:
            r['type'] = 'integer'
    elif field_desc.type in _FLOAT_TYPES:
        if field_desc.label == FD.LABEL_REPEATED:
            r['type'] = 'array'
            r['items'] = {'type': 'number'}
        else:
            r['type'] = 'number'
    elif field_desc.type in _STRING_TYPES:
        if field_desc.label == FD.LABEL_REPEATED:
            r['type'] = 'array'
            r['items'] = {'type': 'string'}
        else:
            r['type'] = 'string'
    elif field_desc.type in _BOOL_TYPES:
        if field_desc.label == FD.LABEL_REPEATED:
            r['type'] = 'array'
            r['items'] = {'type': 'boolean'}
        else:
            r['type'] = 'boolean'
    elif field_desc.type in _STRUCT_TYPES:
        if field_desc.label == FD.LABEL_REPEATED:
            r['type'] = 'array'
            r['items'] = infer_schema(field_desc.message_type)
        else:
            r = infer_schema(field_desc.message_type)
    return r, validation_options


def infer_schema(descriptor):
    """
    Extracts jsonschema from protobuf message descriptor (.DESCRIPTOR attribute).

    :param descriptor: Protobuf message descriptor to inspect.
    :rtype: dict
    """
    properties = {}
    required = []
    r = {
        'type': 'object',
        'additionalProperties': True,
        'properties': properties
    }
    # Extract message options
    for field_desc, value in descriptor.GetOptions().ListFields():
        if field_desc.name == 'title':
            r['title'] = value
        elif field_desc.name == 'description':
            r['description'] = value
    for field_desc in descriptor.fields:
        field_schema, field_options = _infer_field_schema(field_desc)
        if field_options.pop('required', False):
            required.append(field_desc.camelcase_name)
        properties[field_desc.camelcase_name] = field_schema
    if required:
        r['required'] = required
    return r
