# -*- coding: utf-8 -*-
"""
Вспомогательный класс для обработки дампов и отправки их в агрегатор
"""
import re
import os
import time
import logging
import tempfile
import subprocess

import gevent
import requests
from sepelib.subprocess.util import terminate

from instancectl.cms import load_dump_json
from instancectl.utils import ensure_path
from instancectl.lib import coreaggrutil
from instancectl.lib.procutil import ExecuteHelper


class MinidumpUtilError(Exception):
    pass


log = logging.getLogger(__name__)

CTYPE_ITAG_PREFIX = 'a_ctype_'
PRJ_ITAG_PREFIX = 'a_prj_'
TEMP_DIR = 'state/tmp/'
PROCESS_CHECK_TIMEOUT = 0.5
SOCKET_TIMEOUT = 5.0


def _run_subprocess(command, stdout=None, stderr=None, timeout=None,
                    exception=MinidumpUtilError, limits=None):
    execute_helper = ExecuteHelper(limits)

    with open(os.devnull, 'a') as devnull:
        process = subprocess.Popen(
            command,
            preexec_fn=execute_helper,
            stdout=stdout or devnull,
            stderr=stderr or devnull,
        )

        deadline = time.time() + timeout
        rc = process.poll()
        while rc is None and time.time() < deadline:
            gevent.sleep(PROCESS_CHECK_TIMEOUT)
            rc = process.poll()

    if rc is None:
        log.error('Command %s timeout: %s seconds, killing it', command, timeout)
        try:
            terminate(process)
        except Exception as err:
            log.error('Cannot kill timed out process', exc_info=True)
            raise exception('Cannot kill timed out process: {}'.format(err))
        else:
            raise exception('Command {} timed out: {} seconds'.format(command, timeout))

    if rc != 0:
        log.error('Command %s fail: exit code %s; will not send minidump to aggregator', command, rc)
        raise exception('Command {} fail: exit code {}'.format(command, rc))


def extract_coredump_traces(file_name, binary, gdb_path, gdb_timeout, limits=None):
    """
        Обработать coredump с помощью gdb и получить stacktrace
        :param file_name: имя файла корки
        :param binary: путь к покорковшемуся бинарнику
        :param gdb_path: путь к gdb
        :param gdb_timeout: таймаут парсинга корки gdb
        :param limits: лимиты для подпроцессов
        :return: текст стектрейса
    """
    ensure_path(TEMP_DIR)

    gdb_command = [
        gdb_path,
        '--core={0}'.format(file_name),
        '--eval-command=thread apply all bt 999',  # CORES-193
        '--batch',
        '--args',
        binary,
    ]

    log.info('Starting gdb: %s', gdb_command)

    with tempfile.TemporaryFile(dir=TEMP_DIR) as gdb_output:
        # Нельзя использовать stdout=PIPE из-за потенциально большого выхлопа gdb,
        # который блокирует процесс. Поэтому дампим во временный файл.
        _run_subprocess(
            gdb_command,
            stdout=gdb_output,
            timeout=gdb_timeout,
            exception=MinidumpUtilError,
            limits=limits
        )

        try:
            gdb_output.seek(0)
            return gdb_output.read()
        except Exception as err:
            log.error('Cannot read gdb stacktrace results', exc_info=True)
            raise MinidumpUtilError('Cannot read minidump_stackwalk results: {}'.format(err))


def get_signal(file_name):
    core_signal = os.environ.get('CORE_SIG', "")
    if not core_signal:
        regexp = re.compile(r".*\.S([0-9]{2}|[0-9]{1})\..*")
        core_signal = regexp.findall(file_name)
        if not core_signal:
            core_signal = ""
        else:
            core_signal = core_signal[0]
    return str(core_signal)


def send_minidump_to_saas(
    file_name,
    parsed_traces,
    url, service,
    instance_name,
    ctype,
    parse_error="",
    parse_result=coreaggrutil.ParseResult.OK,
):
    """
        Отправка минидампа/корки в SaaS агрегатор
        :param file_name: имя минидампа или корки
        :param parsed_traces: текст трейсов для аггрегации
        :param url: URL для запроса
        :param service: имя покорковшегося сервиса
        :param instance_name: имя покорковшегося инстанса
        :param ctype: ctype покорковшегося инстанса
        :param parse_error: текст с сообщением об ошибке, если упал gdb
        :param parse_result: FAILED, если gdb упал в процессе работы, OK иначе
    """
    try:
        minidump_mtime = os.path.getmtime(file_name)
    except Exception as err:
        log.info('Cannot get mtime of file %s: %s', file_name, err)
        raise MinidumpUtilError('Cannot get mtime of file {}: {}'.format(file_name, err))

    core_signal = get_signal(file_name)

    try:
        json_dump = load_dump_json()
    except Exception:
        log.exception("Cannot read dump.json")
        json_dump = {}

    try:
        if not coreaggrutil.is_new_cores_aggregator(url):
            response = requests.post(
                url,
                data=parsed_traces,
                params={
                    'time': int(minidump_mtime),
                    'service': service,
                    'ctype': ctype or '',
                    'server': instance_name,
                    'signal': core_signal,
                },
                timeout=SOCKET_TIMEOUT
            )
            response.raise_for_status()

            coredumps_url = "https://coredumps.n.yandex-team.ru/submit_core"
            log.info("Also send core to cores-testing")

            data = {"parsed_traces": parsed_traces, "dump_json": json_dump}
            requests.post(
                coredumps_url,
                json=data,
                params={
                    'time': int(minidump_mtime),
                    'service': service,
                    'ctype': ctype or '',
                    'server': instance_name,
                    'nanny_service_id': os.environ.get("NANNY_SERVICE_ID", ""),
                    'signal': core_signal,
                    'parse_error': parse_error,
                    'parse_result': parse_result,
                    'old_cores': True
                },
                timeout=SOCKET_TIMEOUT,
            )

        else:
            data = {"parsed_traces": parsed_traces, "dump_json": json_dump}
            response = requests.post(
                url,
                json=data,
                params={
                    'time': int(minidump_mtime),
                    'service': service,
                    'ctype': ctype or '',
                    'server': instance_name,
                    'nanny_service_id': os.environ.get("NANNY_SERVICE_ID", ""),
                    'signal': core_signal,
                    'parse_error': parse_error,
                    'parse_result': parse_result,
                    'old_cores': False,
                },
                timeout=SOCKET_TIMEOUT,
            )
            response.raise_for_status()
    except Exception as e:
        raise MinidumpUtilError('Cannot send minidump to SaaS aggregator: {}'.format(e))
    log.info('Minidumps aggregator response: %s', response.text)
