from __future__ import unicode_literals

import logging

import porto
import porto.exceptions
import six
from yp_proto.yp.client.hq.proto import types_pb2
from gevent import lock
from instancectl import constants
from instancectl import errors

log = logging.getLogger('portoutil')


def escape_porto_env_var_value(v):
    if isinstance(v, str):
        try:
            v = v.decode('utf8')
        except Exception:
            raise errors.ContainerError('Non-UTF8 strings (including secrets) cannot be used in env variable values')
    return v.replace('\\', '\\\\').replace(';', '\\;')


def cast_env_to_porto_string(env):
    """
    :type env: dict
    :rtype: unicode
    """
    return '; '.join('{}={}'.format(k, escape_porto_env_var_value(v)) for k, v in six.iteritems(env))


def cast_ulimits_to_porto_string(limits):
    """
    Returns ulimits string "nproc: <soft_limit> <hard_limit>; nofile: <soft_limit> <hard_limit>; ..."

    :type limits: dict[unicode, instancectl.common.RLimit]
    :rtype: unicode
    """
    return '; '.join('{}: {} {}'.format(l, r.soft, r.hard) for l, r in six.iteritems(limits))


def cast_resource_allocation_to_porto_dict(a):
    """
    :type a: clusterpb.types_pb2.ResourceAllocation
    :rtype: dict[unicode, unicode]
    """
    r = {}
    # There is no validation here, assuming it has been validated earlier
    for l in a.limit:
        if l.name == 'mem':
            r['memory_limit'] = six.text_type(l.scalar.value)
        elif l.name == 'cpu':
            r['cpu_limit'] = six.text_type(l.scalar.value / 1000.0) + 'c'
        elif l.name == 'net':
            r['net_limit'] = six.text_type(l.scalar.value)
        elif l.name == 'io':
            r['io_limit'] = six.text_type(l.scalar.value)
    for l in a.request:
        if l.name == 'mem':
            r['memory_guarantee'] = six.text_type(l.scalar.value)
        elif l.name == 'cpu':
            r['cpu_guarantee'] = six.text_type(l.scalar.value / 1000.0) + 'c'
        elif l.name == 'net':
            r['net_guarantee'] = six.text_type(l.scalar.value)
        elif l.name == 'io':
            r['io_guarantee'] = six.text_type(l.scalar.value)
    return r


class PortoConnectionContext(object):
    COMMUNICATION_ERROR_TYPES = (
        porto.exceptions.SocketError,
        porto.exceptions.SocketTimeout,
    )

    def __init__(self):
        # Sockets are gevent monkeypatched here so we can use ordinary socket, not gevent.socket
        self.c = porto.Connection(lock_constructor=lock.Semaphore)

    def __enter__(self):
        """
        :rtype: porto.api.Connection
        """
        self.c.connect()
        return self.c

    def __exit__(self, exc_type, exc_val, exc_tb):
        try:
            self.c.disconnect()
        except Exception:
            log.error('Porto connection error: cannot disconnect', exc_info=True)
        if exc_type in self.COMMUNICATION_ERROR_TYPES:
            raise errors.ContainerCommunicationError('Porto communication error: {}'.format(exc_val))
        if exc_type is porto.exceptions.UnknownError:
            raise errors.ContainerUnknownError('Porto unknown error: {}'.format(exc_val))
        if exc_type is porto.exceptions.ContainerDoesNotExist:
            raise errors.ContainerNotFoundError('Porto container not found: {}'.format(exc_val))
        if exc_type and issubclass(exc_type, Exception):
            raise errors.ContainerError('Porto container error: {}'.format(exc_val))


def _get_int_porto_property_or_none(c, p):
    """
    :type c: porto.api.Connection
    :type p: unicode
    :rtype: int | None
    """
    try:
        v = c.GetProperty('self', p)
    except Exception:
        return None
    try:
        return int(v)
    except ValueError:
        return None


def _get_cpu_porto_property_or_none(c, p):
    """
    :type c: porto.api.Connection
    :type p: unicode
    :rtype: int | None
    """
    try:
        v = c.GetProperty('self', p)
    except Exception:
        return None
    v = v.rstrip('c')
    try:
        return int(float(v) * 1000)
    except ValueError:
        return None


def make_current_resources_allocation():
    """
    :rtype: types_pb2.ResourceAllocation
    """
    a = types_pb2.ResourceAllocation()
    with PortoConnectionContext() as c:
        mem_limit = _get_int_porto_property_or_none(c, 'memory_limit_total')
        mem_guarantee = _get_int_porto_property_or_none(c, 'memory_guarantee_total')
        # We MUST have CPU limit on iss_hook_start/iss_hook_install container in APP_CONTAINER mode
        # cpu_limit = TOTAL_CPU_COUNT otherwise, but assuming here that we are in APP_CONTAINER mode
        # Similar assumption is made for cpu_guarantee.
        cpu_limit = _get_cpu_porto_property_or_none(c, 'cpu_limit')
        cpu_guarantee = _get_cpu_porto_property_or_none(c, 'cpu_limit')
    if mem_limit:
        l = a.limit.add()
        l.name = 'mem'
        l.type = types_pb2.ComputeResource.SCALAR
        l.scalar.value = mem_limit
    if mem_guarantee:
        l = a.request.add()
        l.name = 'mem'
        l.type = types_pb2.ComputeResource.SCALAR
        l.scalar.value = mem_guarantee
    if cpu_limit:
        l = a.limit.add()
        l.name = 'cpu'
        l.type = types_pb2.ComputeResource.SCALAR
        l.scalar.value = cpu_limit
    if cpu_guarantee:
        l = a.request.add()
        l.name = 'cpu'
        l.type = types_pb2.ComputeResource.SCALAR
        l.scalar.value = cpu_guarantee
    return a


def make_app_container_allocation(a):
    """
    :type a: types_pb2.ResourceAllocation
    :rtype: types_pb2.ResourceAllocation
    """
    rv = types_pb2.ResourceAllocation()
    for l in a.limit:
        if l.name == 'mem':
            r = rv.limit.add()
            r.CopyFrom(l)
            r.scalar.value -= constants.INSTANCECTL_MEM_LIMIT_RESERVED
        elif l.name == 'cpu':
            r = rv.limit.add()
            r.CopyFrom(l)
            r.scalar.value -= constants.INSTANCECTL_CPU_LIMIT_RESERVED
    for l in a.request:
        if l.name == 'mem':
            r = rv.limit.add()
            r.CopyFrom(l)
            r.scalar.value -= constants.INSTANCECTL_MEM_GUARANTEE_RESERVED
        elif l.name == 'cpu':
            r = rv.limit.add()
            r.CopyFrom(l)
            r.scalar.value -= constants.INSTANCECTL_CPU_GUARANTEE_RESERVED
    return rv
