from __future__ import unicode_literals

import logging
import signal
from collections import namedtuple

import porto

from instancectl.lib import portoutil
from . import interfaces

PortoMode = namedtuple('PortoMode', ('enabled', 'isolate', 'virt_mode'))


class VirtMode(object):
    OS = 'os'
    APP = 'app'


def cast_command_to_str(cmd):
    """
    :type cmd: list[str | unicode]
    """
    # https://github.com/yandex/porto/blob/master/src/property.cpp#L1205
    rv = []
    for arg in cmd:
        escaped = arg.replace("'", "'\\''")
        r = "'{}'".format(escaped)
        rv.append(r)
    # RTCSUPPORT-4139: /sbin/init is special case
    if len(rv) == 1 and rv[0] == "'/sbin/init'":
        return "/sbin/init"
    return ' '.join(rv)


class PortoContainer(interfaces.IProcess):

    WAIT_DELAY = 0.5

    PORTO_PROPERTY_COMMAND = 'command'
    PORTO_PROPERTY_ISOLATE = 'isolate'
    PORTO_PROPERTY_USER = 'user'
    PORTO_PROPERTY_ENV = 'env'
    PORTO_PROPERTY_LIMITS = 'ulimit'
    PORTO_PROPERTY_CAPABILITIES_AMBIENT = 'capabilities_ambient'
    PORTO_PROPERTY_ENABLE_PORTO = 'enable_porto'
    PORTO_PROPERTY_OOM_IS_FATAL = 'oom_is_fatal'
    PORTO_PROPERTY_STDOUT_PATH = 'stdout_path'
    PORTO_PROPERTY_STDERR_PATH = 'stderr_path'
    PORTO_PROPERTY_CWD = 'cwd'
    PORTO_PROPERTY_VIRT_MODE = 'virt_mode'
    PORTO_PROPERTY_NET = 'net'
    PORTO_PROPERTY_CORE_COMMAND = 'core_command'
    PORTO_PROPERTY_CPU_POLICY = 'cpu_policy'

    PORTO_DATA_ROOT_PID = 'root_pid'
    PORTO_DATA_EXIT_STATUS = 'exit_status'
    PORTO_DATA_STATE = 'state'
    PORTO_DEAD = 'dead'

    def __init__(self, sub_container, command, stdout, stderr, env, limits, cwd, security_context, allocation,
                 porto_mode, core_command, cpu_policy=None):
        """
        :type sub_container: str | unicode
        :type command: list[str | unicode]
        :type stdout: str | unicode
        :type stderr: str | unicode
        :type env: dict
        :type limits: dict
        :type security_context: clusterpb.types_pb2.SecurityContext
        :type allocation: clusterpb.types_pb2.ResourcesAllocation
        :type porto_mode: PortoMode
        :type core_command: str | unicode
        :type cpu_policy: str | unicode | None
        """
        self._sub_container = sub_container
        self._name = 'self/{}'.format(sub_container)
        self._command = cast_command_to_str(command)
        self._stdout = stdout
        self._stderr = stderr
        self._env = env
        self._limits = limits
        self._log = logging.getLogger('porto-container[{}]'.format(self._sub_container))
        self._cwd = cwd
        self._security_context = security_context
        self._allocation = allocation
        self._porto_mode = porto_mode
        self.pid = None
        self._core_command = core_command
        self._cpu_policy = cpu_policy

    def _set(self, c, key, value):
        """
        :type c: porto.api.Connection
        :type key: str | unicode
        :type value: str | unicode
        """
        self._log.info('Set property %s: %r', key, value)
        if isinstance(value, unicode):
            value = value.encode('utf8')
        c.SetProperty(self._name, key, value)

    def _destroy_container_ignore(self, c):
        """
        Destroys container if it exists.

        :type c: porto.api.Connection
        """
        try:
            c.Destroy(self._name)
        except porto.exceptions.ContainerDoesNotExist:
            pass
        else:
            self._log.info('Container destroyed')

    def start(self):
        """
        * Destroys existing container if it exists
        * Creates container
        * Executes command in container
        """
        with portoutil.PortoConnectionContext() as c:
            try:
                self._create(c)
                self._start(c)
            except Exception:
                self._destroy_container_ignore(c)
                raise
            self._log.info('Container started')

    def get_exit_status(self):
        with portoutil.PortoConnectionContext() as c:
            s = c.GetData(self._name, self.PORTO_DATA_STATE)
            if s != self.PORTO_DEAD:
                return None
            try:
                exit_status = c.GetData(self._name, self.PORTO_DATA_EXIT_STATUS)
            except porto.exceptions.InvalidState:
                return None
            else:
                return int(exit_status)

    def wait(self, timeout):
        with portoutil.PortoConnectionContext() as c:
            r = c.Wait([self._name], timeout * 1000)
        if r == self._name:
            return self.get_exit_status()
        return None

    def terminate(self):
        with portoutil.PortoConnectionContext() as c:
            c.Kill(self._name, signal.SIGTERM)

    def kill(self):
        with portoutil.PortoConnectionContext() as c:
            c.Destroy(self._name)

    def create(self):
        with portoutil.PortoConnectionContext() as c:
            self._create(c)

    def _create(self, c):
        """
        :type c: porto.api.Connection
        """
        self._destroy_container_ignore(c)
        c.Create(self._name)
        self._log.info('Container created')
        self._set(c, self.PORTO_PROPERTY_COMMAND, self._command)
        self._set(c, self.PORTO_PROPERTY_STDOUT_PATH, self._stdout)
        self._set(c, self.PORTO_PROPERTY_STDERR_PATH, self._stderr)
        self._set(c, self.PORTO_PROPERTY_ENABLE_PORTO, self._security_context.porto_access_policy)
        self._set(c, self.PORTO_PROPERTY_ISOLATE, self._porto_mode.isolate)
        self._set(c, self.PORTO_PROPERTY_VIRT_MODE, self._porto_mode.virt_mode)
        self._set(c, self.PORTO_PROPERTY_NET, 'inherited')
        self._set(c, self.PORTO_PROPERTY_OOM_IS_FATAL, 'false')
        # Temporary disable setting cpu_policy: SWAT-8134
        # if self._cpu_policy:
        #    self._set(c, self.PORTO_PROPERTY_CPU_POLICY, self._cpu_policy)
        if self._env:
            e = portoutil.cast_env_to_porto_string(self._env)
            self._set(c, self.PORTO_PROPERTY_ENV, e)
        if self._limits:
            l = portoutil.cast_ulimits_to_porto_string(self._limits)
            self._set(c, self.PORTO_PROPERTY_LIMITS, l)
        if self._cwd:
            self._set(c, self.PORTO_PROPERTY_CWD, self._cwd)
        if self._security_context.capabilities:
            caps = ';'.join(self._security_context.capabilities)
            self._set(c, self.PORTO_PROPERTY_CAPABILITIES_AMBIENT, caps)
        if self._security_context.run_as_user:
            self._set(c, self.PORTO_PROPERTY_USER, self._security_context.run_as_user)

        resource_props = portoutil.cast_resource_allocation_to_porto_dict(self._allocation)
        if resource_props:
            for k, v in resource_props.iteritems():
                self._set(c, k, v)
        if self._core_command:
            self._set(c, self.PORTO_PROPERTY_CORE_COMMAND, self._core_command)

    def _start(self, c):
        """
        :type c: porto.api.Connection
        """
        c.Start(self._name)
        try:
            pid = c.GetData(self._name, self.PORTO_DATA_ROOT_PID)
        except porto.exceptions.InvalidState:
            self.pid = None
        else:
            self.pid = int(pid)

