from __future__ import unicode_literals

import os
import resource
import subprocess
import time

from instancectl.config.defaults import run_limit_types


class ExtendedPopen(subprocess.Popen):
    """
    pass_fds backport from python 3.2
    """
    def __init__(self, cmd, *args, **kwargs):
        self.exit_status = None
        subprocess.Popen.__init__(self, cmd, *args, **kwargs)

    def get_exit_status(self):
        self.poll()
        return self.exit_status

    def _handle_exitstatus(self, sts, *args, **kwargs):
        # We need to get exit original exit status. To do it WITHOUT
        # gevent monkey patch we must override this strange method
        self.exit_status = sts
        return super(ExtendedPopen, self)._handle_exitstatus(sts, *args, **kwargs)

    def _on_child(self, watcher):
        # We need to get exit original exit status. To do it WITH
        # gevent monkey patch we must override this strange method
        watcher.stop()
        status = watcher.rstatus
        self.exit_status = status
        if os.WIFSIGNALED(status):
            self.returncode = -os.WTERMSIG(status)
        else:
            self.returncode = os.WEXITSTATUS(status)
        self.result.set(self.returncode)


class ExecuteHelper(object):
    """
    Will run in subprocess after fork.

    Sets rlimits in subprocess. We use it to control if coredump
    must be created.
    """

    def __init__(self, limits):
        """
        :type limits: dict[unicode, instancectl.common.RLimit]
        """
        self._limits = limits

    def __call__(self):
        for name, limit in self._limits.iteritems():
            try:
                resource.setrlimit(run_limit_types[name], (limit.soft, limit.hard))
            except Exception:
                pass


class ScriptExecutionContext(object):

    def __init__(self, timeout):
        """
        :type timeout: float | types.NoneType
        """
        if timeout is None:
            self.deadline = None
        else:
            self.deadline = time.time() + timeout

    def is_timed_out(self):
        """
        :rtype: bool
        """
        if self.deadline is None:
            return False
        return time.time() > self.deadline


class ScriptRunner(object):

    PROCESS_CHECK_TIMEOUT = 0.1

    def __init__(self, exec_runner):
        """
        :type exec_runner: instancectl.lib.handler.exec_runner.ExecRunner
        """
        self.exec_runner = exec_runner

    def run(self, script, arguments, timeout, additional_env):
        """
        :param script: script body
        :type script: unicode
        :type arguments: list[unicode]
        :type timeout: int | types.NoneType
        :type additional_env: dict[unicode, unicode]
        :rtype: int
        """
        command = ['/bin/sh', '-c', script] + arguments
        return self.exec_runner.run(command, timeout, additional_env)
