from __future__ import unicode_literals

import httplib
import logging

import flask
import json as ujson

from instancectl.lib import pbutil
from instancectl.lib import jsonschemautil
from . import status_pb2
from . import parse_request
from . import exceptions


PROTOBUF_CONTENT_TYPE = 'application/x-protobuf'
JSON_CONTENT_TYPE = 'application/json'
JSON_CONTENT_TYPES = (JSON_CONTENT_TYPE,)
PROTOBUF_CONTENT_TYPES = (PROTOBUF_CONTENT_TYPE, 'application/octet-stream',)
MIME_MATCHES = JSON_CONTENT_TYPES + PROTOBUF_CONTENT_TYPES

BAD_METHOD_ERROR = status_pb2.Status(status="Failure", message='Method not supported', code=httplib.BAD_REQUEST)


def make_response(protobuf_object, accept_mimetypes, status=httplib.OK):
    """
    Transforms protobuf message to flask response
    according to accept_types (from flask request) setting provided HTTP status.

    :param protobuf_object: protobuf message to serialize and put in HTTP body
    :param accept_mimetypes: accept_mimetypes attribute of flask request
    :param status: HTTP status to set

    :return: flask Response object
    """
    match = accept_mimetypes.best_match(MIME_MATCHES, default=JSON_CONTENT_TYPE)
    if match in PROTOBUF_CONTENT_TYPES:
        body = protobuf_object.SerializeToString()
        content_type = PROTOBUF_CONTENT_TYPE
    else:
        # If we don't use including_default_value_fields empty lists
        # won't be present in result javascript, which can unexpected and will require
        # complicated handling on receiving side.
        try:
            d = pbutil.pb_to_jsondict(protobuf_object)
        except Exception as e:
            d = pbutil.pb_to_jsondict(status_pb2.Status(code=httplib.INTERNAL_SERVER_ERROR,
                                                        status="Failure",
                                                        message=str(e)))
        body = ujson.dumps(d)  # Use ujson to serialize response - it's faster
        content_type = JSON_CONTENT_TYPE
    return flask.Response(body, status=status, content_type=content_type)


def call_user_handler(handler, flask_request, protobuf_request, auth_subject, log=None):
    """
    Calls user RPC handler and catches exceptions, transforming them into RPC response.

    :param handler: user handler to call
    :param flask_request: flask Request object
    :param protobuf_request: protobuf request
    :param auth_subject: authentication subject information if authentication enabled or None
    :param log: logger to report exceptions to
    :return: flask response object
    """
    try:
        response = handler(protobuf_request, auth_subject)
    except exceptions.RpcError as e:
        if e.status == httplib.INTERNAL_SERVER_ERROR:
            log.exception('Request processing failed.')
        return make_response(status_pb2.Status(status="Failure", code=e.status, message=str(e)),
                             flask_request.accept_mimetypes, status=e.status)
    except Exception as e:
        log.exception('Request processing failed.')
        return make_response(status_pb2.Status(
                status="Failure",
                code=httplib.INTERNAL_SERVER_ERROR,
                message=str(e)
        ), flask_request.accept_mimetypes, status=httplib.INTERNAL_SERVER_ERROR)
    return make_response(response, flask_request.accept_mimetypes)


def parse_flask_request(protobuf_request, flask_request, log):
    method = flask_request.method
    if method == 'GET':
        try:
            parse_request.init_from_args(protobuf_request, flask_request.args)
        except Exception as e:
            log.exception('Failed to parse request to "%s"', flask_request.path)
            return make_response(status_pb2.Status(
                    status="Failure",
                    code=httplib.BAD_REQUEST,
                    message=str(e)
            ), flask_request.accept_mimetypes, status=httplib.BAD_REQUEST)
        return
    elif method == 'POST':
        try:
            parse_request.init_from_content(protobuf_request,
                                            flask_request.get_data(),
                                            flask_request.content_type)
        except Exception as e:
            log.exception('Failed to parse request to "%s"', flask_request.path)
            return make_response(status_pb2.Status(
                    status="Failure",
                    code=httplib.BAD_REQUEST,
                    message=str(e)
            ), flask_request.accept_mimetypes, status=httplib.BAD_REQUEST)
        return
    return make_response(BAD_METHOD_ERROR, flask_request.accept_mimetypes, status=httplib.BAD_REQUEST)


class HttpRpcBlueprint(flask.Blueprint):
    """
    Blueprint which provides utilities for protobuf-over-http RPC services.
    """

    _default_methods = ('GET', 'POST')
    # We register all methods in flask so that we can manually check actual method
    # and return proper API response.
    # Not quite sure about HEAD and OPTIONS.
    ALL_METHODS = ('GET', 'POST', 'PUT', 'DELETE', 'PATCH')

    def __init__(self, name, import_name, url_prefix):
        super(HttpRpcBlueprint, self).__init__(name, import_name, url_prefix=url_prefix)
        self.log = logging.getLogger(name)
        self.route('/', endpoint='schema', methods=('GET',))(self._schema_ctrl)
        self.rpc_methods = []

    def _schema_ctrl(self):
        return flask.Response(ujson.dumps(self.rpc_methods), status=200, mimetype=JSON_CONTENT_TYPE)

    def method(self, method_name, request_type, response_type, allow_http_methods=_default_methods):
        """
        Decorator to register provided function as RPC handler.

        Upon invocation user function will be called as
        >> function(protobuf_request, auth_subject)
        Where:
            * protobuf_request - protobuf message object, which was instantiated from :param request_type:
            * auth_subject - Object holding authentication info, particular login or
                             None (if authentication was disabled).
        Function should either:
            * return protobuf response, which must be of type :param response_type:
            * raise one of RpcErrors, defined in .exceptions.py

        Either way response or exception will be serialized in JSON/Protobuf depending on
        Accept: HTTP header. Default content type for response is JSON.

        :param method_name: RPC method name.
        :param request_type: Protobuf message, which holds incoming parameters.
        :param response_type: Protobuf message, which will contain response.
        :param allow_http_methods: A list of allow HTTP methods. POST and GET by default.
        """
        request_schema = jsonschemautil.infer_schema(request_type.DESCRIPTOR)
        self.rpc_methods.append((
            method_name,
            request_schema,
            jsonschemautil.infer_schema(response_type.DESCRIPTOR),
        ))

        def real_method(function):
            def handle_http_request():
                flask_request = flask.request
                if flask_request.method not in allow_http_methods:
                    return make_response(BAD_METHOD_ERROR,
                                         flask_request.accept_mimetypes,
                                         httplib.BAD_REQUEST)

                # Initialize empty protobuf object
                protobuf_request = request_type()
                # Parse request
                error_response = parse_flask_request(protobuf_request, flask_request, self.log)
                if error_response is not None:
                    return error_response
                # Disable authentication
                auth_subject = None
                # Call user handler
                return call_user_handler(function, flask_request, protobuf_request, auth_subject, log=self.log)

            url = '/' + method_name + '/'
            # Disable legacy authentication mechanism, we have our own
            handle_http_request.need_auth = False
            handle_http_request.handler = function
            # Register flask handler
            return self.route(url, endpoint=method_name, methods=self.ALL_METHODS)(handle_http_request)

        return real_method
