"""
Utilities to parse request from URL query or body content (JSON or Protobuf).
"""
from __future__ import unicode_literals

import json as ujson
import google.protobuf.descriptor
import google.protobuf.json_format as json_format
import google.protobuf.message

from instancectl.lib import pbutil

FD = google.protobuf.descriptor.FieldDescriptor

_INT_TYPES = (
    FD.TYPE_INT32,
    FD.TYPE_INT64,
    FD.TYPE_SINT32,
    FD.TYPE_SINT64,
    FD.TYPE_FIXED32,
    FD.TYPE_FIXED64,
    FD.TYPE_SFIXED32,
    FD.TYPE_SFIXED64,
)
_FLOAT_TYPES = (
    FD.TYPE_FLOAT,
    FD.TYPE_DOUBLE
)
_NUM_TYPES = _INT_TYPES + _FLOAT_TYPES
_STRING_TYPES = (
    FD.TYPE_STRING,
)
_BOOL_TYPES = (
    FD.TYPE_BOOL,
)
_STRUCT_TYPES = (
    FD.TYPE_MESSAGE,
)

LABEL_REPEATED = FD.LABEL_REPEATED


def _set_field_value(pb_object, field_desc, value):
    if field_desc.label == LABEL_REPEATED:
        getattr(pb_object, field_desc.name).append(value)
    else:
        setattr(pb_object, field_desc.name, value)


def _set_attribute(pb_object, k, v):
    attr_name, _, cdr = k.partition('.')
    # Protobuf rules of converting JSON -> Protobuf -> JSON
    # define that keys must be camel case.
    # E.g. field foo_bar in JSON is converted to fooBar.
    # So we use fields_by_camelcase_name to get field description.
    # Check type of this field in descriptor
    field_desc = pb_object.DESCRIPTOR.fields_by_camelcase_name.get(attr_name)
    # Seems like extra field specified - ignore it
    if field_desc is None:
        return
    if field_desc.type in _INT_TYPES:
        if cdr:  # There is leftover, but field type is primitive
            raise ValueError('{} has no attribute {}'.format(pb_object.DESCRIPTOR.name, k))
        _set_field_value(pb_object, field_desc, int(v))
        return
    if field_desc.type in _STRING_TYPES:
        if cdr:  # There is leftover, but field type is primitive
            raise ValueError('{} has no attribute {}'.format(pb_object.DESCRIPTOR.name, k))
        _set_field_value(pb_object, field_desc, v)
        return
    if field_desc.type in _BOOL_TYPES:
        if cdr:  # There is leftover, but field type is primitive
            raise ValueError('{} has no attribute {}'.format(pb_object.DESCRIPTOR.name, k))
        # Only accept true or false to match official protobuf->JSON mapping.
        if v == 'true':
            value = True
        elif v == 'false':
            value = False
        else:
            raise ValueError('Wrong value "{}" for bool field'.format(v))
        _set_field_value(pb_object, field_desc, value)
        return
    if field_desc.type in _FLOAT_TYPES:
        if cdr:  # There is leftover, but field type is primitive
            raise ValueError('{} has no attribute {}'.format(pb_object.DESCRIPTOR.name, k))
        _set_field_value(pb_object, field_desc, float(v))
    if field_desc.type in _STRUCT_TYPES:
        # We do not support repeated fields at the moment
        if field_desc.label == LABEL_REPEATED:
            raise ValueError('{} is repeated field, cannot set via query'.format(field_desc.name))
        if not cdr:  # There is no leftover, but field is composite
            raise ValueError('{} is message field, cannot set directly'.format(field_desc.name))
        _set_attribute(getattr(pb_object, field_desc.name), cdr, v)


def _format_attr_path(prefix, field_name, index=None):
    if not prefix:
        if index is None:
            return field_name
        return '{}[{}]'.format(field_name, index)
    rv = '.'.join((prefix, field_name))
    if index is not None:
        rv += '[{}]'.format(index)
    return rv


def init_from_args(pb_object, args):
    """
    Initialize provided protobuf object from query arguments.

    :param pb_object: Protobuf object to initialize
    :param args: Dictionary with query string
    :type args: werkzeug.datastructures.MultiDict
    """
    for k, v in args.items(multi=True):
        _set_attribute(pb_object, k, v)
    pbutil.validate_pb_schema(pb_object)


def init_from_content(protobuf_object, content, content_type="application/json"):
    """
    Initialize provided protobuf object from request body content.

    :param protobuf_object: Protobuf object to initialize
    :param content: String with body content
    :param content_type: Type of content encoding
    """
    if not isinstance(content, str):
        raise TypeError('Unsupported content to load: {}, need str'.format(type(content)))
    if 'application/json' in content_type:
        # Use ujson for faster load times
        try:
            d = ujson.loads(content)
        except ValueError as e:
            raise ValueError("Failed to parse request: {}".format(e.message))

        try:
            json_format.ParseDict(d, protobuf_object)
        except json_format.ParseError as e:
            raise ValueError(e.message)
        pbutil.validate_pb_schema(protobuf_object)
        return
    if content_type in ('application/x-protobuf', 'application/octet-stream',):
        try:
            protobuf_object.MergeFromString(content)
        except google.protobuf.message.DecodeError as e:
            raise ValueError(e.message)
        pbutil.validate_pb_schema(protobuf_object)
        return
    raise ValueError('Not supported content-type: {}'.format(content_type))
