from __future__ import unicode_literals

import logging

import six
from yp_proto.yp.client.hq.proto import types_pb2
from sepelib.util import retry
from instancectl import constants
from instancectl import errors
from instancectl.clients.vault import errors as vault_errors
from instancectl.clients.yav import errors as yav_errors

log = logging.getLogger('specutil')


def expand_command(c, env):
    """
    :type c: Iterable
    :type env: dict[unicode, unicode]
    """
    t = list(c)
    for i, v in enumerate(t):
        try:
            c[i] = v.format(**env)
        except KeyError as e:
            n = str(e).strip("'")
            log.error('Cannot expand command: variable "%s" not found. It means that you have string {%s} in your '
                      'instance spec. See docs: https://nda.ya.ru/3UKD2D '
                      'Full command: "%s", expanding failed on: "%s"', n, n, t, v)
            raise
        except IndexError:
            log.error('Cannot expand instance spec command. '
                      'It means that you have string string like "{1}" in your instance spec. '
                      'See docs: https://nda.ya.ru/3UKD2D Full command: "%s", expanding failed on: "%s"', t, v)
            raise


def expand_handler_params(h, env):
    expand_command(h.exec_action.command, env)
    h.tcp_socket.host = h.tcp_socket.host.format(**env)
    h.tcp_socket.port = h.tcp_socket.port.format(**env)
    h.http_get.host = h.http_get.host.format(**env)
    h.http_get.port = h.http_get.port.format(**env)


def expand_revision_spec_variables(s, env):
    """
    :type s: clusterpb.types_pb2.InstanceRevision
    :type env: dict[unicode, unicode]
    """
    for h in s.notify_action.handlers:
        expand_handler_params(h, env)


def expand_container_spec_variables(s, env):
    """
    :type s: clusterpb.types_pb2.Container
    :type env: dict[unicode, unicode]
    """
    logging.info('Expanding container "%s" command', s.name)
    expand_command(s.command, env)
    for h in s.readiness_probe.handlers:
        logging.info('Expanding container "%s" readiness check', s.name)
        expand_handler_params(h, env)
    logging.info('Expanding container "%s" pre_stop action', s.name)
    expand_handler_params(s.lifecycle.pre_stop, env)
    if s.HasField('reopen_log_action'):
        logging.info('Expanding container "%s" reopen_log action', s.name)
        expand_handler_params(s.reopen_log_action.handler, env)


def populate_container_env(s, env, vault_client, yav_client):
    """
    :type s: clusterpb.types_pb2.Container
    :type env: dict[unicode, unicode]
    :type vault_client: instancectl.clients.vault.client.VaultClient
    :type yav_client: instancectl.clients.yav.client.YavClient
    """
    log.info('Populating env values')
    for v in s.env:
        if v.value_from.type == types_pb2.EnvVarSource.SECRET_ENV:
            try:
                result = vault_client.get_secret_field(secret=v.value_from.secret_env.keychain_secret,
                                                       field=v.value_from.secret_env.field)
            except vault_errors.VaultError as e:
                raise errors.ContainerSpecProcessError('Cannot get secret from Nanny Vault: {}'.format(e))
            env[v.name] = result.encode('utf-8')
        elif v.value_from.type == types_pb2.EnvVarSource.VAULT_SECRET_ENV:
            try:
                result = yav_client.get_secret_field(secret=v.value_from.vault_secret_env.vault_secret,
                                                     field=v.value_from.vault_secret_env.field)
            except yav_errors.YavError as e:
                raise errors.ContainerSpecProcessError('Cannot get secret from YaVault: {}'.format(e))
            env[v.name] = result.encode('utf-8')
        elif v.value_from.type == types_pb2.EnvVarSource.LITERAL_ENV:
            env[v.name] = v.value_from.literal_env.value.encode('utf-8')


def is_restart_needed(restart_policy, exit_status):
    """
    :type restart_policy: clusterpb.types_pb2.RestartPolicy
    :type exit_status: clusterpb.types_pb2.ExitStatus
    :rtype: bool
    """
    if restart_policy.type == types_pb2.RestartPolicy.ALWAYS:
        return True
    if restart_policy.type == types_pb2.RestartPolicy.NEVER:
        return False
    if restart_policy.type == types_pb2.RestartPolicy.ON_FAILURE:
        if exit_status.if_exited and exit_status.exit_status == 0:
            return False
    return True


def prepare_security_context(c, env):
    """
    :type c: clusterpb.types_pb2.SecurityContext
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    """
    if not c.capabilities and env.node_name != env.hostname:
        c.capabilities.append(constants.CAP_NET_BIND_SERVICE)
    if not c.porto_access_policy:
        c.porto_access_policy = 'true'


def prepare_restart_policy(p):
    """
    :type p: clusterpb.types_pb2.RestartPolicy
    """
    if not p.min_period_seconds:
        p.min_period_seconds = constants.DEFAULT_CONTAINER_RESTART_MIN_DELAY
    if not p.max_period_seconds:
        p.max_period_seconds = constants.DEFAULT_CONTAINER_RESTART_MAX_DELAY
    if not p.period_backoff:
        p.period_backoff = constants.STATUS_UPDATE_RESTART_PERIOD_BACKOFF
    if not p.period_jitter_seconds:
        p.period_jitter_seconds = constants.DEFAULT_CONTAINER_RESTART_MAX_JITTER
    return p


def prepare_handler(h, env):
    if h.type == types_pb2.Handler.TCP_SOCKET:
        if not h.tcp_socket.host:
            h.tcp_socket.host = env.hostname
        if not h.tcp_socket.port:
            h.tcp_socket.port = six.text_type(env.instance_port)
    if h.type == types_pb2.Handler.HTTP_GET:
        if not h.http_get.host:
            h.http_get.host = env.hostname
        if not h.http_get.port:
            h.http_get.port = six.text_type(env.instance_port)
        if not h.http_get.uri_scheme:
            h.http_get.uri_scheme = 'HTTP'


def prepare_readiness_probe(p, env):
    """
    :type p: clusterpb.types_pb2.Probe
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    """
    if not p.initial_delay_seconds:
        p.initial_delay_seconds = constants.STATUS_UPDATE_INITIAL_DELAY_SECONDS
    if not p.min_period_seconds:
        p.min_period_seconds = constants.STATUS_UPDATE_MIN_RESTART_PERIOD
    if not p.max_period_seconds:
        p.max_period_seconds = constants.STATUS_UPDATE_MAX_RESTART_PERIOD
    if not p.period_backoff:
        p.period_backoff = constants.STATUS_UPDATE_RESTART_PERIOD_BACKOFF
    for h in p.handlers:
        prepare_handler(h, env)
    return p


def prepare_lifecycle(l, env):
    if not l.stop_grace_period_seconds:
        l.stop_grace_period_seconds = constants.DEFAULT_CONTAINER_TERMINATE_TIMEOUT
    if not l.termination_grace_period_seconds:
        l.termination_grace_period_seconds = constants.DEFAULT_CONTAINER_KILL_TIMEOUT
    prepare_handler(l.pre_stop, env)


def prepare_volume(v):
    if v.type == types_pb2.Volume.ITS:
        if not v.name:
            v.name = 'controls'
        if not v.its_volume.its_url:
            v.its_volume.its_url = constants.DEFAULT_ITS_URL
        if not v.its_volume.period_seconds:
            v.its_volume.period_seconds = constants.ITS_PERIOD
        if not v.its_volume.max_retry_period_seconds:
            v.its_volume.max_retry_period_seconds = constants.ITS_RETRY_MAX_PERIOD


def prepare_notify_action(n, env):
    """
    :type n: clusterpb.types_pb2.NotifyAction
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    """
    for h in n.handlers:
        prepare_handler(h, env)


def prepare_instance_spec(s, env):
    """
    :type s: clusterpb.types_pb2.InstanceRevision
    :type env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype: clusterpb.types_pb2.InstanceRevision
    """
    for c in s.container:
        prepare_restart_policy(c.restart_policy)
        prepare_readiness_probe(c.readiness_probe, env)
        prepare_lifecycle(c.lifecycle, env)
        prepare_handler(c.reopen_log_action.handler, env)
        prepare_security_context(c.security_context, env)
    for c in s.init_containers:
        prepare_restart_policy(c.restart_policy)
        prepare_security_context(c.security_context, env)
        c.restart_policy.type = types_pb2.RestartPolicy.ON_FAILURE
    for v in s.volume:
        prepare_volume(v)
    prepare_notify_action(s.notify_action, env)
    return s


def create_restart_sleeper(spec):
    return retry.RetrySleeper(
        delay=spec.restart_policy.min_period_seconds,
        max_delay=spec.restart_policy.max_period_seconds,
        backoff=spec.restart_policy.period_backoff,
        max_jitter=spec.restart_policy.period_jitter_seconds,
        max_tries=-1
    )
