from __future__ import unicode_literals

import argparse
import errno
import logging
import os
import socket
import sys

from sepelib.util.exc import format_exc

from instancectl import common
from instancectl import errors
from instancectl import monkey
from instancectl import utils
from instancectl.cmd import log as setup_logging
from instancectl.version import VERSION
import instancectl.cmd.call_reopenlog
import instancectl.cmd.get_status
import instancectl.cmd.print_status
import instancectl.cmd.install
import instancectl.cmd.notify
import instancectl.cmd.send_minidump
import instancectl.cmd.start
import instancectl.cmd.stop_instance
import instancectl.cmd.uninstall
import instancectl.cmd.core_process


monkey.patch_all()

log = logging.getLogger('main')


def parse_args(argv):
    # Fix UnicodeDecodeError on accidental non-ASCII symbol typing in command line
    argv = [arg.decode("utf-8") for arg in argv]

    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--version', action='version', version=VERSION)
    parser.add_argument('--verbose', action='store_true', help='be verbose')
    parser.add_argument('--console', action='store_true', help='force logging to stdout')

    subparsers = parser.add_subparsers(title='Possible actions')

    # Main actions
    instancectl.cmd.install.add_parsers(subparsers)
    instancectl.cmd.start.add_parsers(subparsers)
    instancectl.cmd.uninstall.add_parsers(subparsers)

    # In-parallel actions
    instancectl.cmd.call_reopenlog.add_parsers(subparsers)
    instancectl.cmd.get_status.add_parsers(subparsers)
    instancectl.cmd.notify.add_parsers(subparsers)
    instancectl.cmd.print_status.add_parsers(subparsers)
    instancectl.cmd.send_minidump.add_parsers(subparsers)
    instancectl.cmd.stop_instance.add_parsers(subparsers)
    instancectl.cmd.core_process.add_parsers(subparsers)

    return parser.parse_args(argv)


def main():
    try:
        # Porto redirects stdout and stderr of root process to /dev/null
        # in "virt_mode: os", so we have to redirect it to files
        # for details see https://wiki.yandex-team.ru/porto/propertiesanddata/virtmode/
        if utils.is_porto_virt_mode_os_root_pid():
            utils.redirect_output_to_files()

        # init log to stdout to log errors
        # if we failed to parse config etc.
        console_handler = setup_logging.setup_logging_to_stderr()

        # bsconfig starts us without any args (in most cases)
        if len(sys.argv) == 1:
            sys.argv.append('start')

        instance_dir = os.environ.get('BSCONFIG_IDIR')

        if instance_dir is None:
            # BSCONFIG_IDIR env var is not present under ISS, but InstanceCtl
            # and user's binaries use BSCONFIG_IDIR env var, so we must set it.
            instance_dir = os.path.dirname(os.path.abspath(sys.argv[0]))

        # bsconfig may start us not in instance dir, chdir to it
        os.chdir(instance_dir)

        args = parse_args(sys.argv[1:])
        args.handle(args, console_handler)

    # some handy exception handling
    # to show nicely formatted and detailed enough
    # error messages to user
    except SystemExit:
        raise
    except KeyboardInterrupt:
        log.info("exiting on user request")
        exit_code = 0
    except errors.InstanceCtlCannotAcquireLock as e:
        log.critical("cannot acquire lock file, probably another copy of instancectl is running already: {}".format(e))
        exit_code = common.INSTANCE_CTL_CANNOT_ACQUIRE_LOCK_EXIT_CODE
    except errors.InstanceCtlInitError as e:
        log.critical("cannot make init actions: {}".format(e), exc_info=True)
        exit_code = common.INSTANCE_CTL_CANNOT_INIT
    except socket.error as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.critical("exiting on socket error: code='{}' msg='{}'".format(errorcode, e.strerror), exc_info=True)
        exit_code = 1
    # IOError has 'filename' attribute so we handle it
    # separately from Exception
    except IOError as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.critical("exiting on io error: file='{}' code='{}' msg='{}'".format(e.filename, errorcode, e.strerror),
                     exc_info=True)
        exit_code = 1
    except Exception as e:
        # BaseException.message is deprecated so we manually
        # construct nice looking (hopefully) string to log
        log.critical(format_exc("exiting on error", e), exc_info=True)
        exit_code = 1
    else:
        exit_code = 0
    raise SystemExit(exit_code)


if __name__ == '__main__':
    main()
