# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

import gevent.event
from instancectl.lib import statusutil
from . import interfaces


class InstanceRevisionStatusCacher(interfaces.IInstanceRevisionStatusCacher):
    """
    Holds instance current status

    https://wiki.yandex-team.ru/jandekspoisk/sepe/instancectl/#status

    :type status: types_pb2.RevisionStatus
    """

    def __init__(self, status, criterion_containers):
        """
        :type criterion_containers: list[unicode]
        """
        super(InstanceRevisionStatusCacher, self).__init__()
        self.log = logging.getLogger('instance-status-cacher')
        self.criterion_containers = criterion_containers
        self.status_changed_event = gevent.event.Event()
        self.install_cond_set = gevent.event.Event()
        self.status = status

    def set_container_status(self, s):
        """
        :type s: clusterpb.types_pb2.ContainerStatus
        :rtype: bool
        """
        updated = False
        for c in self.status.container:
            if c.name == s.name:
                if not statusutil.are_container_statuses_equal(c, s):
                    c.CopyFrom(s)
                    updated = True
        if not self.criterion_containers:
            # We must not set ready = True if there is no criterion containers
            return False
        if statusutil.update_rev_status(self.status, self.criterion_containers):
            self.log.info('New instance status: %s', statusutil.format_status(self.status))
            self.status_changed_event.set()
        return updated

    def set_ready_condition(self, c):
        """
        :type c: clusterpb.types_pb2.Condition
        :rtype: bool
        """
        old = self.status.ready.status
        self.status.ready.CopyFrom(c)
        if self.status.ready.status != old:
            self.log.info('New instance status: %s', statusutil.format_status(self.status))
            self.status_changed_event.set()
            return True
        return False

    def set_installed_condition(self, c):
        """
        :type c: clusterpb.types_pb2.Condition
        :rtype: bool
        """
        old = self.status.installed.status
        self.status.installed.CopyFrom(c)
        self.install_cond_set.set()
        if self.status.installed.status != old:
            self.log.info('New instance status: %s', statusutil.format_status(self.status))
            self.status_changed_event.set()
            return True
        return False

    def get_status(self):
        """
        Возвращает статус инстанса, вычисленный последний раз.

        :rtype: types_pb2.RevisionStatus
        """
        self.install_cond_set.wait()
        return self.status
