# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

import gevent
from requests import exceptions
from sepelib.util import retry

from instancectl import constants
from instancectl.lib import statusutil
from instancectl.clients.unixsocket_rpc import client
from infra.nanny.instancectl.proto import instancectl_pb2
from infra.nanny.instancectl.proto import instancectl_stub
from instancectl.utils import get_event_logger


log = logging.getLogger('actions.status')
event_log = get_event_logger('status')


class InstanceStatusCheckError(Exception):
    pass


class InstanceNotReadyError(Exception):

    def __init__(self, status, *args):
        super(InstanceNotReadyError, self).__init__(*args)
        self.status = status


class Attempt(object):
    __slots__ = ['attempt']

    def __init__(self):
        self.attempt = 0

    def increment(self):
        self.attempt += 1
        return self.attempt


class ExternalInstanceStatusChecker(object):

    RETRY_EXCEPTIONS = (InstanceStatusCheckError, InstanceNotReadyError, gevent.Timeout)

    def __init__(self, url, max_tries, delay, backoff, max_delay, req_timeout):
        self.url = url
        self.status_check_tries = max_tries
        self.request_timeout = req_timeout
        s = retry.RetrySleeper(delay=delay,
                               backoff=backoff,
                               max_delay=max_delay,
                               max_tries=max_tries - 1)  # Workaround sepelib bug
        self.retry = retry.RetryWithTimeout(attempt_timeout=req_timeout,
                                            retry_sleeper=s,
                                            retry_exceptions=self.RETRY_EXCEPTIONS)

    def _get_instance_status(self, attempt):
        """
        :type attempt: Attempt
        """
        log.info('Checking instance status attempt %s/%s', attempt.increment(), self.status_check_tries)
        rpc = client.UnixsocketRpcClient(rpc_url=self.url, request_timeout=self.request_timeout)
        stub = instancectl_stub.InstanceRevisionServiceStub(rpc)
        req = instancectl_pb2.GetInstanceRevisionStatusRequest()
        try:
            resp = stub.get_instance_revision_status(req)
        except exceptions.ConnectionError as e:
            # При невозможности сделать connect на unixsocket для получения состояния инстанса,
            # не следует логгировать трейс, т.к. ISS в ряде ситуаций запускает iss_hook_status
            # при незапущенном iss_hook_start, в таких ситуациях не нужно пугать пользователя трейсами в логах.
            log.warning('Cannot check instance status: cannot connect to InstanceCtl on unixsocket: %s', e)
            raise InstanceStatusCheckError('Cannot connect to InstanceCtl via unixsocket: {}'.format(e))
        except (gevent.Timeout, Exception) as e:
            log.exception('Cannot check instance status')
            event_log.error('Cannot check instance status: %s', e)
            raise InstanceStatusCheckError('Cannot get instance status: {}'.format(e))
        else:
            status = resp.status
            log.info('Instance status check result: %s', statusutil.format_status(status))
            msg = '; '.join(['"{}": {}'.format(s.name, statusutil.format_status(s)) for s in status.container])
            log.info('Container statuses: %s', msg)
            if status.ready.status == constants.CONDITION_TRUE:
                return status
            raise InstanceNotReadyError(status, 'Instance is not ready')

    def get_instance_status(self):
        """
        Получает статус инстанса, делая запрос в unixsocket с повторными попытками

        :rtype: bool
        """
        return self.retry(self._get_instance_status, attempt=Attempt())
