from __future__ import unicode_literals

import logging
import time

import gevent
from yp_proto.yp.client.hq.proto import types_pb2

from sepelib.gevent import greenthread

from instancectl.lib import statusutil
from instancectl import constants
from infra.nanny.instancectl.proto import instancectl_pb2
from .. import container_status_evaluator


class ContainerStatusUpdater(greenthread.GreenThread):
    """
    Periodically starts container status computation.
    """

    ITERATION_DELAY = 1.0

    def __init__(self, job, spec, cacher):
        """
        :type job: instancectl.jobs.job.Job
        :type spec: clusterpb.types_pb2.Probe
        :type cacher: instancectl.status.cacher.InstanceRevisionStatusCacher
        """
        super(ContainerStatusUpdater, self).__init__()
        self.log = logging.getLogger('status-updater_{}'.format(job.name))
        self.job = job
        self.spec = spec
        self.cacher = cacher
        self.evaluator = container_status_evaluator.ContainerStatusEvaluator(spec, job)
        self.sleeper = statusutil.make_sleeper_from_probe(spec)

    def container_running_long_enough(self):
        """
        :rtype: float
        """
        s = self.job.get_state()
        start_time = s.last_start_time.ToMilliseconds()
        current = time.time() * 1000
        return (current - start_time) > self.spec.initial_delay_seconds * 1000

    def is_container_started(self):
        """
        :rtype: bool
        """
        s = self.job.get_state()
        return s.state == instancectl_pb2.ContainerInternalState.STARTED

    def _set_status_terminated(self):
        s = self._make_terminated_status()
        self.cacher.set_container_status(s)

    def _run(self):
        self.log.info('Waiting for container to be started')
        self.job.start_event.wait()
        self.job.start_event.clear()
        self.job.termination_event.clear()
        self.log.info('Waiting %s seconds before status check', self.spec.initial_delay_seconds)
        gevent.sleep(self.spec.initial_delay_seconds)
        if not self.is_container_started() or not self.container_running_long_enough():
            self._set_status_terminated()
            return
        while True:
            if self.update_status():
                self.log.info('Status changed, reset check sleeper')
                self.sleeper.reset()
            t = self.sleeper.get_next_time_to_sleep()
            self.log.info('Waiting %s seconds before status check', t)
            if self.job.termination_event.wait(t):
                self._set_status_terminated()
                return

    def _make_terminated_status(self):
        """
        :rtype: clusterpb.types_pb2.ContainerStatus
        """
        job_state = self.job.get_state()
        c = types_pb2.ContainerStatus()
        c.name = self.job.name
        c.restart_count = job_state.restart_count
        c.last_termination_status.CopyFrom(job_state.last_termination_status)
        c.ready.status = constants.CONDITION_FALSE
        c.ready.reason = 'ContainerNotReady'
        c.ready.last_transition_time.GetCurrentTime()
        i = self.evaluator.make_installed_condition(job_state)
        c.installed.CopyFrom(i)
        return c

    def update_status(self):
        self.log.info('Start status computation')
        s = self.evaluator.make_container_status()
        self.log.info('New status: %s', statusutil.format_status(s))
        return self.cacher.set_container_status(s)

    def run(self):
        s = self._make_terminated_status()
        self.log.info('Set initial status: %s', statusutil.format_status(s))
        self.cacher.set_container_status(s)
        while True:
            try:
                self._run()
            except Exception:
                self.log.exception('Status updating fail')
                pass
            gevent.sleep(self.ITERATION_DELAY)
