# coding=utf-8
from __future__ import unicode_literals

import errno
import fcntl
import glob
import logging
import logging.handlers
import os
import sys
import tempfile

from sepelib.util import fs

import instancectl.config.errors


PORTO_VIRT_MODE_OS_ROOT_PID = 1
STDOUT_FILENAME = 'instancectl.out'
STDERR_FILENAME = 'instancectl.err'
log = logging.getLogger(__file__)


def lock(fd):
    log = logging.getLogger('lock')
    try:
        fcntl.lockf(fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except (IOError, OSError) as e:
        log.debug('Failed locking %s: %r' % (fd.name, e))
        return False

    log.debug('Locked %s' % fd.name)
    return True


class FileLock(object):
    """
    Работа с взятым файллоком.
    """
    def __init__(self, filename):
        self._filename = filename
        self._fd = None

    def __enter__(self):
        dir_path = os.path.dirname(self._filename)
        fs.makedirs_ignore(dir_path)
        self._fd = open(self._filename, 'a')
        if lock(self._fd):
            return self
        else:
            self._fd.close()
            raise EnvironmentError('Failed locking {}'.format(self._filename))

    def __exit__(self, exc_type, exc_val, exc_tb):
        self._fd.close()


def is_fresh_core_exists(core_mask, binary_mtime, core_files_limit, coredumps_dir):
    """
    Returns True if:
      * Too many files matching :param core_mask: present in :param coredumps_dir:;
      * Or at least one fresh core matching :param core_mask: exists.

    :type binary_mtime: int | float
    :type core_mask: unicode
    :type coredumps_dir: unicode
    :type core_files_limit: int
    :rtype: bool
    """
    if not os.path.exists(coredumps_dir):
        return False

    cores = glob.glob(os.path.join(coredumps_dir, core_mask))

    if not cores:
        return False

    if len(cores) >= core_files_limit:
        # Don't create too many cores even if cores don't refer to the binary
        return True

    for core in cores:
        core_mtime = os.stat(core).st_mtime
        if core_mtime > binary_mtime:
            return True

    # All cores are older than binary
    return False


def tryopen(node, mode):
    log = logging.getLogger('tryopen')
    if not node is None:
        try:
            return open(node, mode)
        except IOError:
            log.debug(
                "Failed oppening file %s with mode %r:",
                node,
                mode,
                exc_info=True,
            )
    return


def detach_from_cms_agent():
    try:
        os.setsid()
    except OSError:
        log.warning('Cannot create new process group (skipping it)')

    redirect_output_to_files()

    with file('/dev/null', 'r') as si:
        os.dup2(si.fileno(), sys.stdin.fileno())


def ensure_path(dir_path):
    """Создаем директорию, если ее нет"""
    try:
        os.makedirs(dir_path)
    except OSError as err:
        if err.errno != errno.EEXIST:
            raise


def redirect_output_to_files():
    """
    Redirects stdout and stderr to files
    """
    sys.stdout.flush()
    sys.stderr.flush()

    with file(STDOUT_FILENAME, 'a') as fd:
        os.dup2(fd.fileno(), sys.stdout.fileno())

    with file(STDERR_FILENAME, 'a') as fd:
        os.dup2(fd.fileno(), sys.stderr.fileno())


def file_write_via_renaming(file_path, contents, temp_dir='/var/tmp/', ensure_paths=True):
    """Запись в файл через переименование временного файла"""

    # не совсем безопасно, т.к. дира может быть удалена сразу после создания
    if ensure_paths:
        ensure_path(temp_dir)

    if isinstance(contents, unicode):
        contents = contents.encode('utf-8')

    with tempfile.NamedTemporaryFile(dir=temp_dir, delete=False) as fd:
        fd.write(contents)
        fd.flush()
        # fsync is required for data consistency
        os.fsync(fd.fileno())
        temp_filename = fd.name

    # не совсем безопасно, т.к. дира может быть удалена сразу после создания
    if ensure_paths:
        ensure_path(os.path.dirname(file_path))

    os.rename(temp_filename, file_path)


def remove_file_if_exists(filename):
    """Удалить файл, если он существует"""
    try:
        os.remove(filename)
    except OSError as err:
        if err.errno != errno.ENOENT:
            raise


EVENTS_LOGGER_NAME = 'events'


def get_event_logger(logger_relative_name=None):
    """
    Возвращает логгер для вывода в пользовательский лог: SWAT-2199

    :type logger_relative_name: unicode | str
    """
    logger_name = 'events'
    if logger_relative_name is not None:
        logger_name = '{}.{}'.format(logger_name, logger_relative_name)
    return logging.getLogger(logger_name)


def parse_config_int(config, name):
    r = config[name]
    try:
        return int(r)
    except ValueError:
        raise instancectl.config.errors.StatusCheckConfigError('Cannot parse int config parameter: %s', name)


def cast_exit_status_to_pb(status_int, status_pb):
    """
    :type status_int: int
    :type status_pb: clusterpb.types_pb2.ExitStatus
    """
    if os.WIFEXITED(status_int):
        status_pb.if_exited = True
        status_pb.exit_status = os.WEXITSTATUS(status_int)
        return
    if os.WIFSIGNALED(status_int):
        status_pb.if_signaled = True
        status_pb.term_signal = os.WTERMSIG(status_int)
        status_pb.coredumped = os.WCOREDUMP(status_int)
        return


def cast_exit_status_pb_to_env(s):
    """
    :type s: clusterpb.types_pb2.ExitStatus
    :rtype: dict
    """
    return {
        'INSTANCECTL_SECTION_EXIT_STATUS': unicode(s.exit_status) if s.if_exited else '',
        'INSTANCECTL_SECTION_TERM_SIGNAL': unicode(s.term_signal) if s.if_signaled else '',
        'INSTANCECTL_SECTION_COREDUMPED': '1' if s.coredumped else ''
    }


def read_last_bytes_from_file(filename, byte_limit):
    """
    :type filename: unicode
    :type byte_limit: int
    :rtype: str
    """
    with open(filename) as fd:
        # fd.seek uses fseek which may read buffer of unknown size: http://bugs.python.org/issue21638
        # we use os.lseek and os.read to avoid it
        fd_int = fd.fileno()
        file_size = os.lseek(fd_int, 0, os.SEEK_END)
        if file_size < byte_limit:
            byte_limit = file_size
        os.lseek(fd_int, -byte_limit, os.SEEK_END)
        return os.read(fd_int, byte_limit)


def is_porto_virt_mode_os_root_pid():
    """
    :rtype: bool
    """
    return os.getpid() == PORTO_VIRT_MODE_OS_ROOT_PID


class StreamToHandler(object):
    """
    Fake file-like stream object that redirects writes to a loghander instance.

    Copied from sepelib/flask/server.py
    """

    def __init__(self, handler):
        self.handler = handler
        self.linebuf = ''

    def isatty(self):
        return False

    def write(self, buf):
        # remove line endings, handler will add them anyway
        buf = buf.rstrip()
        record = logging.makeLogRecord({'msg': buf,
                                        'levelno': logging.DEBUG})
        self.handler.handle(record)


def make_exc(exc_type, msg):
    """
    :param exc_type: desired exception class
    :type msg: unicode
    :rtype: tuple
    """
    exc_info = sys.exc_info()
    raise exc_type, msg, exc_info[2]
