# -*- coding: utf-8 -*-
from __future__ import unicode_literals

try:
    import argparse
    import errno
    import logging
    import logging.handlers
    import os
    import sys
    import socket

    from sepelib.util.log import setup_logging_to_stdout, setup_logging_to_file, create_handler_from_config
    from sepelib.util.exc import format_exc
    from sepelib.core import config

    from its_client.poller import ItsPoller
    from its_client.version import VERSION
except KeyboardInterrupt:
    # don't bother user with python stack trace
    # if interrupted (by Ctrl+C) during imports
    raise SystemExit(1)


log = logging.getLogger('main')


def _parse_args(argv):
    """
    Command line parsing utility.
    """
    parser = argparse.ArgumentParser(description='ITS client')
    parser.add_argument('-v', '--version',
                        action='version',
                        version=VERSION)
    parser.add_argument('-c', '--cfg',
                        default=None,
                        action='store', help='path to config file')
    # --console is used to dump logs during debugging with IDE
    # when stdout isn't terminal but we don't want to log to file
    parser.add_argument('--console',
                        default=False,
                        action='store_true',
                        help='redirect log to stdout (overrides config)')
    config.augment_args_parser(parser)
    return parser.parse_args(argv)


def setup_logging(logcfg, console_handler, console=False):
    """
    Setup logs according to config
    """
    # disable requests module logging
    logging.getLogger('requests').setLevel(logging.WARNING)

    if not sys.stdout.isatty() and not console:
        handler = create_handler_from_config(logcfg)
        logger = setup_logging_to_file(handler)
        # don't forget to remove stdout handler
        logging.getLogger().removeHandler(console_handler)
    else:
        logger = console_handler
    # python2.6 doesn't automagically converts level name to int
    # so we do it manually
    loglevel = logging.getLevelName(logcfg.get('loglevel', 'INFO'))
    logger.setLevel(loglevel)


def main(argv):
    try:
        # init log to stdout to log errors
        # if we failed to parse config etc.
        console_handler = setup_logging_to_stdout()
        # process command line arguments
        args = _parse_args(argv[1:])
        # load config
        config.load(args.cfg)
        # log configuration part
        logcfg = config.get_value('log')
        setup_logging(logcfg, console_handler, console=args.console)
        # log what config we're using
        # couldn't do that earlier
        # because log has been just set up
        if args.cfg:
            log.info("using '{0}' as config".format(args.cfg))
        else:
            log.info("using hardwired config")

        # create and run application
        cfg = config.get_value('its_client')
        application = ItsPoller.from_config(cfg)
        application.run()
    # some handy exception handling
    # to show nicely formatted and detailed enough
    # error messages to user
    except SystemExit:
        raise
    except KeyboardInterrupt:
        log.info("exiting on user request")
        exit_code = 0
    except socket.error as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on socket error: code='{}' msg='{}'".format(errorcode, e.strerror))
        exit_code = 1
    # IOError has 'filename' attribute so we handle it
    # separately from Exception
    except IOError as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on io error: file='{}' code='{}' msg='{}'".format(
            e.filename, errorcode, e.strerror))
        exit_code = 1
    except Exception as e:
        # BaseException.message is deprecated so we manually
        # construct nice looking (hopefully) string to log
        log.exception(format_exc("exiting on error", e))
        exit_code = 1
    else:
        exit_code = 0
    raise SystemExit(exit_code)
