# -*- coding=utf-8 -*-
"""
Логика получения ручек из ITS
"""
from __future__ import unicode_literals

import os
import socket
import time
import errno
import logging

import gevent
import gevent.event

from sepelib.util.retry import RetrySleeper
from sepelib.util.fs import atomic_write, remove_ignore, makedirs_ignore
from sepelib.yandex.its import ItsClient, ItsControls, ItsApiRequestException
from sepelib.gevent import greenthread


LOGGER_NAME = 'its-client'


log = logging.getLogger(LOGGER_NAME)


def _read_ignore_errors(path):
    try:
        with open(path) as fd:
            return fd.read()
    except EnvironmentError:
        log.exception('Cannot read file %s contents', path)
        return None


def _write_ignore_errors(path, contents):
    log.info('Writing "%s" to %s', contents, path)
    try:
        atomic_write(path, contents)
    except Exception:
        log.exception('Cannot not write "%s" to %s', contents, path)
        return False
    else:
        return True


def _makedirs_ignore_errors(path):
    try:
        makedirs_ignore(path)
    except OSError:
        log.exception('Cannot create dir %s', path)


def _listdir_ignore_errors(path):
    try:
        return os.listdir(path)
    except OSError as e:
        log.info('Cannot list files in %s: %s', path, e)


class ItsPoller(greenthread.GreenThread):
    """
    Поллер ручек из ITS по заданному списку itag'ов

    * После старта ходит в ITS с экспоненциально растущим таймаутом до первого получения ручек
    * После получения ручек от ITS идёт за ними повторно через max-age секунд, которые получает от ITS
      в заголовке Cache-Control
    * При появлении файла refresh.flag сбрасывает таймаут хождения в ITS и идёт за ручками с экспоненциально
      растущим таймаутом до первого успешного получения ручек
    * Сохраняет получаемые ручки в директорию с инстансом и shared_dir
    * При старте демона заменяет ручки инстанса ручками из shared_dir, если последние новее первых,
      это может быть полезно при старте новой конфигурации, для которой еще ни разу не сходили в ITS
    """

    MAIN_THREAD_RESTART_DELAY = 1.0
    CHECK_REFRESH_FLAG_TIMEOUT = 0.5

    DEFAULT_REQ_TIMEOUT = 30.0
    MINIMAL_POLL_TIMEOUT = 5.0
    DEFAULT_POLL_TIMEOUT = 60.0
    MAXIMAL_POLL_TIMEOUT = 300.0
    DEFAULT_MAX_TIMEOUT_JITTER = 5.0
    DEFAULT_REFRESH_FLAG_PATH = './refresh.flag'

    META_DIRNAME = '.its_client'
    ETAG_FILENAME = 'etag'
    VERSION_FILENAME = 'version'

    @classmethod
    def from_config(cls, config):
        return cls(**config)

    def __init__(self, url, itags, controls_dir, shared_dir, refresh_flag, req_timeout, poll_timeout, max_poll_timeout,
                 max_timeout_jitter, service_id=None):
        """
        :type url: str | unicode
        :type service_id: str | unicode | None
        :type itags: list[str | unicode]
        """
        super(ItsPoller, self).__init__()
        self.service_id = service_id
        self.itags = itags
        self.refresh_flag = refresh_flag or self.DEFAULT_REFRESH_FLAG_PATH
        self.poll_timeout = poll_timeout
        self.max_poll_timeout = max_poll_timeout
        self.controls_dir = controls_dir
        self.shared_dir = shared_dir
        self.req_timeout = req_timeout
        self._its_client = ItsClient(url=url, req_timeout=self.req_timeout)
        self.controls_requested_event = gevent.event.Event()
        self.max_timeout_jitter = (max_timeout_jitter if max_timeout_jitter is not None
                                   else self.DEFAULT_MAX_TIMEOUT_JITTER)

    def run(self):
        """
        Основной цикл получения ручек из ITS.
        """
        while True:
            try:
                self._run()
            except (Exception, gevent.Timeout):
                log.exception('ITS poller thread crashed')
            gevent.sleep(self.MAIN_THREAD_RESTART_DELAY)

    def _run(self):
        force_ask = True

        timeout_cleared = True
        sleeper = RetrySleeper(delay=self.MINIMAL_POLL_TIMEOUT, max_delay=self.max_poll_timeout)
        its_controls = self._get_saved_controls(self.controls_dir)

        # при старте извлекаем значения ручек из общего хранилища
        # и сохраняем их в директорию инстанса, если они свежее имеющихся там
        if self.shared_dir:
            shared_controls = self._get_saved_controls(self.shared_dir)
            if shared_controls.version is not None:
                if its_controls.version is None or shared_controls.version > its_controls.version:
                    self._save_its_controls(its_controls, shared_controls, self.controls_dir)
                    its_controls = shared_controls

        while True:
            # идем за ручками в ITS
            log.debug('Polling new controls from ITS')
            old_its_controls = its_controls
            tags = {
                'i': self.itags,
            }
            if self.service_id is not None:
                tags['f'] = [self.service_id]
            try:
                with gevent.Timeout(self.req_timeout):
                    its_controls = self._its_client.get_controls(tags, its_controls.etag, force_ask)
            except (ItsApiRequestException, gevent.Timeout, socket.error):
                log.exception('Cannot poll ITS controls values')
                if not timeout_cleared:
                    sleeper = RetrySleeper(delay=self.MINIMAL_POLL_TIMEOUT, max_delay=self.max_poll_timeout)
                    timeout_cleared = True
            else:
                log.debug('Got ITS controls: %s', its_controls)
                timeout_cleared = False
                sleeper = RetrySleeper(delay=self.poll_timeout,
                                       max_delay=self.max_poll_timeout,
                                       max_jitter=self.max_timeout_jitter)
                force_ask = False

                # ITS может вернуть Not-Modified, в этом случае controls is None
                if its_controls.controls is not None:
                    success = self._save_its_controls(old_its_controls, its_controls, self.controls_dir)
                    if self.shared_dir:
                        success &= self._save_its_controls(old_its_controls, its_controls, self.shared_dir)
                    force_ask = not success

            self.controls_requested_event.set()

            next_poll_time = time.time() + sleeper.get_next_time_to_sleep()

            # проверяем наличие флага refresh.flag, если он появился,
            # сразу идем в ITS, если нет -- идем по стандартному таймауту
            while time.time() < next_poll_time:
                time.sleep(self.CHECK_REFRESH_FLAG_TIMEOUT)
                try:
                    os.unlink(self.refresh_flag)
                except OSError as e:
                    if e.errno != errno.ENOENT:
                        log.exception('Failed deleting refresh.flag')
                else:
                    timeout_cleared = True
                    sleeper = RetrySleeper(delay=self.MINIMAL_POLL_TIMEOUT, max_delay=self.max_poll_timeout)
                    break

    def clear_controls_cache_time(self):
        """
        Сбрасываем время до следующего поллинга ручек из ITS
        """
        try:
            open(self.refresh_flag, 'w').close()
        except EnvironmentError:
            log.exception('Cannot create %s to force ITS polling', self.refresh_flag)
            return False
        else:
            return True

    def _save_its_controls(self, old_its_controls, its_controls, controls_dir):
        """
        Сохраняет значения ручек и их версию в указанную директорию
        (для каждой ручки отдельный файл с ее именем, в который записываем ее значение).

        Если привести хранилище ручек в соответствие полученным ручкам удалось, возвращаем True,
        иначе False

        :type old_its_controls: ItsControls
        :type its_controls: ItsControls
        :type controls_dir: str | unicode
        :rtype: bool
        """
        success = True
        _makedirs_ignore_errors(controls_dir)

        # ETag может меняться даже если реальное содержимое ручек не изменилось: SWAT-3175
        # Чтобы не обновлять mtime файлов ручек, не нужно их перезаписывать, если содержимое
        # осталось прежним, а ETag изменился. Для этого сравниваем закодированное представление
        # версий ручек, зашитое в тело ETag'а, и переписываем ручки только если оно изменилось
        if self._need_update_controls_content(old_its_controls, its_controls):
            for key, value in its_controls.controls.iteritems():
                log.info('Saving ITS control "%s=%s" to %s', key, value, controls_dir)
                success &= _write_ignore_errors(os.path.join(controls_dir, key), value)

        # удаляем лишние файлы из директории с ручками
        control_names = _listdir_ignore_errors(controls_dir) or []
        old_controls = set(control_names).difference(its_controls.controls)
        for key in old_controls:
            control_path = os.path.join(controls_dir, key)
            if os.path.isfile(control_path):
                log.info('Removing "%s" from %s', key, controls_dir)
                try:
                    remove_ignore(control_path)
                except OSError:
                    success = False
                    log.exception('Cannot remove "%s" from %s', key, controls_dir)

        meta_dir = os.path.join(controls_dir, self.META_DIRNAME)
        _makedirs_ignore_errors(meta_dir)

        _write_ignore_errors(os.path.join(meta_dir, self.ETAG_FILENAME), its_controls.etag)
        _write_ignore_errors(os.path.join(meta_dir, self.VERSION_FILENAME), its_controls.version)

        return success

    def _get_saved_controls(self, controls_dir):
        """
        Извлекает ранее сохраненные значения ITS ручек

        :type controls_dir: str | unicode
        :rtype ItsControls:
        """
        log.info('Getting saved ITS controls from %s', controls_dir)
        control_files = _listdir_ignore_errors(controls_dir)
        if control_files is None:
            return ItsControls(controls={}, cache_time=None, etag=None, version=None)

        controls = {}
        for filename in control_files:
            control_filename = os.path.join(controls_dir, filename)

            if os.path.isfile(control_filename):
                value = _read_ignore_errors(control_filename)
                if value is not None:
                    controls[filename] = value

        meta_dir = os.path.join(controls_dir, self.META_DIRNAME)
        etag = _read_ignore_errors(os.path.join(meta_dir, self.ETAG_FILENAME))
        version = _read_ignore_errors(os.path.join(meta_dir, self.VERSION_FILENAME))

        return ItsControls(controls=controls, cache_time=None, etag=etag, version=version)

    @staticmethod
    def _need_update_controls_content(current, new):
        """
        :type current: ItsControls
        :type new: ItsControls
        """
        if current.etag is None or new.etag is None:
            return True

        if ':' not in current.etag or ':' not in new.etag:
            return True

        _, current_version = current.etag.split(':', 1)
        _, new_version = new.etag.split(':', 1)

        return current_version != new_version
