# -*- coding: utf-8 -*-
import click

from infra.swatconf import Signal, Graphic, Text, Alert, Panel, setup, FlapOptions, NotificationOptions


COUNT_SUFFIX = '-count_summ'
TIMER_SUFFIX = '-timer_hgram'
default_flaps = FlapOptions(critical=300, stable=60)

CRITNANNY = NotificationOptions('on_status_change', {
    'status': "CRIT",
    'method': ["telegram"],
    'login': ['ferenets', 'nanny-monitorings']
})

YELLOW = '#db0'
ORANGE = '#fa0'
DARK_ORANGE = '#c80'
GREEN = '#193'
DARK_GREEN = '#041'
RED = '#f23'
DARK_RED = '#700'


def create_nanny_api_alerts_panel():
    panel = Panel('nanny-api-alerts')

    for location in ('sas', 'man', 'vla'):
        panel.append(Alert('nanny-requests-{}'.format(location), Signal('balancer_report-report-https-succ_summ',
                                                                           tag='itype=balancer;ctype=prestable,prod;prj=nanny;geo={}'.format(location)),
                           warn=[None, None], crit=[0, 0],
                           title='{} Success requests count'.format(location.upper())).with_juggler_check(notifications=[CRITNANNY]).update())

    panel.upsert()


def create_replication_ctl_panel(ctype='prod'):
    panel = Panel((ctype + '-' if ctype != 'prod' else '') + 'replication-ctl')

    panel.append(Graphic([Signal('unistat-replication-ctl-eviction-requested-total_axxx',
                                 title='Total requested')]))

    activities = [Signal('unistat-replication-ctl-acknowledge-eviction_summ',
                         title='[MOVE] Acknowledge eviction'),
                  Signal('unistat-replication-ctl-acknowledge-maintenance-move_summ',
                         title='[MOVE] Acknowledge maintenance', color=YELLOW),
                  Signal('unistat-replication-ctl-acknowledge-maintenance-replace_summ',
                         title='[REPLACE] Acknowledge maintenance', color=ORANGE),
                  Signal('unistat-replication-ctl-acknowledge-maintenance-replace-manual_summ',
                         title='[REPLACE/manual] Acknowledge maintenance', color=DARK_ORANGE),
                  Signal('unistat-replication-ctl-create-replica_summ',
                         title='[REPLACE] Create replica', color=GREEN),
                  Signal('unistat-replication-ctl-create-replica-manual_summ',
                         title='[REPLACE/manual] Create replica', color=DARK_GREEN),
                  Signal('unistat-replication-ctl-remove-replicated-pod_summ',
                         title='[REPLACE] Remove replicated pod', color=RED),
                  Signal('unistat-replication-ctl-remove-replicated-pod-manual_summ',
                         title='[REPLACE/manual] Remove replicated pod', color=DARK_RED)
                  ]
    panel.append(Graphic(activities, title='Activities'))

    panel.append(Graphic([Signal('unistat-replication-ctl-replication-wait-service-settle_axxx',
                                 title='Waiting for service to be settled'),
                          Signal('unistat-replication-ctl-replication-broken-pods_axxx',
                                 title='Invalid service settings'),
                          Signal('unistat-replication-ctl-replication-wait-old-replicas-deallocation_axxx',
                                 title='Waiting for old replica deallocation'),
                          Signal('unistat-replication-ctl-replication-wait-rate-limit_axxx',
                                 title='Waiting for rate limit'),
                          Signal('unistat-replication-ctl-replication-wait-replication-budget_axxx',
                                 title='Waiting for disruption budget')], title='Eviction block reasons', stacked=True))

    orly_tag = 'itype=rtcsmall;prj=orly;tier=nanny-production-replication-ctl-awacs-balancer-service'
    panel.append(Graphic([Signal('round(hsum(unistat-handle_operation_request_forbidden_dhhh))', title='Forbidden', tag=orly_tag),
                          Signal('round(hsum(unistat-handle_operation_request_allow_dhhh))', title='Allowed', tag=orly_tag),
                          ], title='Orly Ops'))

    panel.append([Alert('replication_ctl_requested_total', title='Eviction requested total'),
                  Alert('move_maintenance_acknowledged', title='Move maintenance acknowledged'),
                  Alert('replace_maintenance_acknowledged', title='Replace maintenance acknowledged'),
                  Alert('manual_replace_maintenance_acknowledged', title='Manual replace maintenance acknowledged')])

    activities_total = Signal('sum(unistat-replication-ctl-acknowledge-eviction_summ, '
                              'unistat-replication-ctl-acknowledge-maintenance-move_summ, '
                              'unistat-replication-ctl-acknowledge-maintenance-replace_summ, '
                              'unistat-replication-ctl-create-replica_summ, '
                              'unistat-replication-ctl-remove-replicated-pod_summ)')
    panel.append(Alert('replication-ctl-activities-total', activities_total, warn=[0, 0], crit=[0, 0],
                       title='Last hour activities count',
                       value_modify={'type': 'summ',
                                     'window': 3600}).with_juggler_check(notifications=[CRITNANNY]).update())

    panel.upsert()


PANELS = [
    'nanny-api-alerts', 'replication-ctl', 'dev-replication-ctl'
]


@click.command()
@click.option('--no-alerts', '-n', is_flag=True)
@click.option('--panel', '-p', multiple=True, type=click.Choice(PANELS))
def main(no_alerts, panel):
    def update_panel(panel_name):
        assert panel_name in PANELS
        if not panel:
            return True
        return panel_name in panel

    # Common setup
    setup(
        juggler_namespace='nanny',
        juggler_hostname='juggler_hostname',
        disable_alerts_updating=no_alerts,
        user='nanny-robot',
        editors=['alonger', 'disafonov', 'ferenets', 'i-dyachkov', 'reddi', 'romanovich']
    )

    if update_panel('nanny-api-alerts'):
        create_nanny_api_alerts_panel()
    if update_panel('replication-ctl'):
        setup(default_tag='itype=replicationctl;ctype=prod;prj=production-nanny-replication-ctl')
        create_replication_ctl_panel()
    if update_panel('dev-replication-ctl'):
        setup(default_tag='itype=nanny;ctype=test;prj=sgcluster-nanny')
        create_replication_ctl_panel(ctype='dev')
