from __future__ import unicode_literals

import uuid
from instancectl.lib import netutil, envutil
from instancectl.hq import helpers as hq_helpers
from instancectl.hq import federated_client
from instancectl.sd import helpers as sd_helpers
from instancectl.sd import client as sd_client

from nanny_rpc_client import RequestsRpcClient
from clusterpb import hq_stub
from clusterpb import hq_pb2

import constant

STR_LOC = {
    'MSK': envutil.Location.MSK,
    'SAS': envutil.Location.SAS,
    'MAN': envutil.Location.MAN,
    'VLA': envutil.Location.VLA
}


def yp_extract_location_from_hostname(fqdn):
    try:
        loc = fqdn.split('.')[1].upper()
    except Exception:
        raise ValueError('Bad FQDN service "{}"'.format(fqdn))

    if loc in STR_LOC:
        return STR_LOC[loc]
    else:
        return envutil.Location.UNKNOWN


def yp_extract_pod_from_hostname(fqdn):
    try:
        return fqdn.split('.')[0]
    except Exception:
        raise ValueError('Bad FQDN service "{}"'.format(fqdn))


def get_hq_object(service, instance, port=None):
    """
    Returns HQ Object

    :rtype: clusterpb.types_pb2.Instance
    """
    if port is None:
        # YP_LITE
        instance_line = '{}@{}'.format(yp_extract_pod_from_hostname(instance), service)
        hq_url = envutil.LOCATION_TO_HQ_DEFAULT_URL_OPTION_MAP[yp_extract_location_from_hostname(instance)]
    else:
        instance_line = '{}:{}@{}'.format(instance, port, service)
        hq_url = envutil.LOCATION_TO_HQ_DEFAULT_URL_OPTION_MAP[envutil.extract_location_from_hostname(instance)]

    client = RequestsRpcClient('{}rpc/instances/'.format(hq_url), request_timeout=10)
    hq_client = hq_stub.InstanceServiceStub(client)
    find_req = hq_pb2.GetInstanceRequest()
    find_req.id = instance_line
    resp = hq_client.get_instance(find_req)
    return resp.instance


def make_template_plugin_context(federated_url, hq_object, revision=None):
    """
    Generate context for jinja template

    :type federated_url: unicode
    :type hq_object: clusterpb.types_pb2.Instance
    :type revision: unicode

    :rtype: dict
    """
    federated = federated_client.FederatedClient(url=federated_url)
    hq_getter = hq_helpers.HqInstancesGetter(federated_client=federated, env=None)
    sd_getter = sd_helpers.SdInstancesGetter(sd_client=sd_client.SdGrpcClient('testing'), env=None)

    context = {}

    context['NANNY_SERVICE_ID'] = hq_object.meta.service_id

    context['CURRENT_REV'] = hq_object.spec.revision[0].id
    context['USER'] = constant.DEFAULT_ISS_USER
    context['ISS'] = '1'

    context['HQ_INSTANCE_SPEC_HASH'] = str(uuid.uuid4())

    # SHARDNAME & SHARDDIR Deprecated.
    context['BSCONFIG_SHARDNAME'] = ''
    context['BSCONFIG_SHARDDIR'] = './'

    context['CONTAINER'] = 'porto'
    context['container'] = 'lxc'

    context['HOSTNAME'] = hq_object.spec.hostname
    context['PORTO_HOST'] = hq_object.spec.node_name
    context['NODE_NAME'] = hq_object.spec.node_name

    context['BSCONFIG_IPORT'] = unicode(hq_object.spec.allocation.port[0].port)
    context['BSCONFIG_IHOST'] = hq_object.spec.node_name.split('.')[0]
    context['BSCONFIG_INAME'] = '{}:{}'.format(context['BSCONFIG_IHOST'], context['BSCONFIG_IPORT'])

    context['HOME'] = '{}/{}_{}_MeGaHaSh'.format(constant.PLACE_ISS_INSTANCE_PATH,
                                                 context['BSCONFIG_IPORT'],
                                                 context['NANNY_SERVICE_ID'])
    context['BSCONFIG_IDIR'] = context['HOME']

    if revision:
        context['CURRENT_REV'] = revision

    current_rev = None
    for rev in hq_object.spec.revision:
        if rev.id == context['CURRENT_REV']:
            current_rev = rev
            break

    if not current_rev:
        raise Exception('Revision {} not found.'.format(context['CURRENT_REV']))

    context['BSCONFIG_ITAGS'] = ' '.join(current_rev.tags)
    context['tags'] = context['BSCONFIG_ITAGS']

    context['get_hq_instances'] = hq_getter.get_instances
    context['get_sd_endpoints'] = sd_getter.get_endpoints
    context['get_sd_pods'] = sd_getter.get_pods

    context['orthogonal_tags'] = envutil.get_orthogonal_tags_dict(current_rev.tags)

    context['resolve_ipv6_addr'] = netutil.resolve_ipv6_addr
    context['resolve_ipv4_addr'] = netutil.resolve_ipv4_addr

    return context
