"""
Exceptions that can be raised by rpc handlers and caught by machinery in rpc blueprint.

Exceptions match those errors defined in protobuf.
"""
from __future__ import unicode_literals


from requests import codes


class HttpError(Exception):
    """
    HTTP-status of response is not OK
    """
    pass


class RpcError(HttpError):
    def __init__(self, message, redirect_url=None):
        super(RpcError, self).__init__(message)
        self.redirect_url = redirect_url

    status = None  # Http status code for this particular error


class InternalError(RpcError):
    """
    Internal error happened.
    """
    status = codes.INTERNAL_SERVER_ERROR


class BadRequestError(RpcError):
    """
    Request is not valid.
    """
    status = codes.BAD_REQUEST


class UnauthenticatedError(RpcError):
    """
    Request doesn't have proper authentication credentials.
    """
    status = codes.UNAUTHORIZED


class ForbiddenError(RpcError):
    """
    Caller does not have permission to perform this action.
    """
    status = codes.FORBIDDEN


class NotFoundError(RpcError):
    """
    Requested entity not found.
    """
    status = codes.NOT_FOUND


class ConflictError(RpcError):
    """
    Action conflicts with current system state (e.g. compare-and-swap failed)
    """
    status = codes.CONFLICT


class TooManyRequestsError(RpcError):
    """
    TooManyRequestsError means that the request itself was valid,
     but the requested service is overwhelmed at this time.

    Retrying the request after some time might succeed.
    """
    status = codes.TOO_MANY_REQUESTS


class BalancerRetriesExceeded(HttpError):
    """
    Balancer max retries exceeded
    """
    pass


class InvalidResponseError(HttpError):
    """
    Invalid response from server
    """
    pass


errors_by_code = {
    codes.INTERNAL_SERVER_ERROR: InternalError,
    codes.BAD_REQUEST: BadRequestError,
    codes.UNAUTHORIZED: UnauthenticatedError,
    codes.FORBIDDEN: ForbiddenError,
    codes.NOT_FOUND: NotFoundError,
    codes.CONFLICT: ConflictError,
    codes.TOO_MANY_REQUESTS: TooManyRequestsError,
}


class RpcClientRetryFailedError(Exception):
    """
    Maximal retries exceeded
    """
    pass
