import time
import uuid

import requests
import six

from . import exceptions
from . import iface
from . import proto
from . import retry_sleeper


PROTOBUF_CONTENT_TYPE = 'application/x-protobuf'


def make_exception_from_response(resp):
    """
    :type resp: requests.models.Response
    :rtype: exceptions.HttpError
    """
    error_protobuf = proto.status_pb2.Status()

    # There is no serialized error in response content, don't try to deserialize
    if resp.headers.get('Content-Type') != PROTOBUF_CONTENT_TYPE:

        if resp.status_code == requests.codes.GATEWAY_TIMEOUT:
            # We can't be sure that it's exactly "balancer max retries exceeded" error.
            # But I've never seen the other reason for this error, so consider this error as
            # "balancer max retries exceeded"
            return exceptions.BalancerRetriesExceeded('Balancer max retries for request processing exceeded, response: '
                                                      '{0}: "{1}"'.format(resp.status_code, resp.content))
        raise RuntimeError('Unsupported response code from service: {0}: "{1}"'.format(resp.status_code, resp.content))

    try:
        error_protobuf.MergeFromString(resp.content)
    except Exception:
        # Failed to load protobuf error response
        raise RuntimeError('Invalid response from server: {0}: "{1}"'.format(resp.status_code, resp.content))

    exc_class = exceptions.errors_by_code.get(error_protobuf.code)
    if exc_class is None:
        raise RuntimeError('Unsupported response code {0} for error content: "{1}"'.format(resp.status_code,
                                                                                           error_protobuf))
    return exc_class(error_protobuf.message, redirect_url=error_protobuf.redirect_url)


class RequestParams(object):
    def __init__(self):
        self._oauth_token = None
        self._session_id = None
        self._request_timeout = None
        self._req_id_header = None
        self._user_agent = None

    def copy(self):
        rv = RequestParams()
        rv.oauth_token = self._oauth_token
        rv.session_id = self._session_id
        rv.request_timeout = self._request_timeout
        rv.req_id_header = self._req_id_header
        rv.user_agent = self._user_agent
        return rv

    def merge(self, kwargs):
        if 'oauth_token' in kwargs:
            self.oauth_token = kwargs['oauth_token']
        if 'session_id' in kwargs:
            self.session_id = kwargs['session_id']
        if 'request_timeout' in kwargs:
            self.request_timeout = kwargs['request_timeout']
        if 'user_agent' in kwargs:
            self.user_agent = kwargs['user_agent']

    @property
    def req_id_header(self):
        return self._req_id_header

    @req_id_header.setter
    def req_id_header(self, value):
        if value is None:
            self._req_id_header = None
            return
        if not isinstance(value, (str, six.text_type)):
            raise TypeError('Value must be str or unicode, got {0}'.format(type(value)))
        self._req_id_header = value

    @property
    def oauth_token(self):
        return self._oauth_token

    @oauth_token.setter
    def oauth_token(self, value):
        if value and self._session_id:
            raise ValueError('Cannot set token, session_id already set. Please choose only one.')
        self._oauth_token = value

    @property
    def session_id(self):
        return self._session_id

    @session_id.setter
    def session_id(self, value):
        if value and self._oauth_token:
            raise ValueError('Cannot set session_id, oauth_token already set. Please choose only one.')
        self._session_id = value

    @property
    def request_timeout(self):
        return self._request_timeout

    @request_timeout.setter
    def request_timeout(self, value):
        if value is None:
            self._request_timeout = None
            return
        if not isinstance(value, (int, float)):
            raise TypeError('Value must be integer or float, got {0}'.format(type(value)))
        if value < .1:
            raise ValueError('Min value is .1, got {0}'.format(value))
        if value > 300:
            raise ValueError('Max value is 300, got {0}'.format(value))
        self._request_timeout = value

    @property
    def user_agent(self):
        return self._user_agent

    @user_agent.setter
    def user_agent(self, value):
        self._user_agent = value


class RequestParamsProcessor(object):

    DEFAULT_HEADERS = {
        'Accept': 'application/x-protobuf',
        'Content-Type': 'application/x-protobuf',
        'User-Agent': 'Python RequestsRpcClient'
    }

    def __init__(self, request_params):
        """
        :type request_params: RequestParams
        """
        self._request_params = request_params

    def prepare_request(self, request_protobuf, kwargs):
        """
        :type kwargs: dict
        :rtype: dict
        """
        if kwargs:
            params = self._request_params.copy()
            params.merge(kwargs)
        else:
            params = self._request_params
        headers = self.DEFAULT_HEADERS.copy()
        # Set authentication headers
        if params.oauth_token:
            headers['Authorization'] = 'OAuth ' + params.oauth_token
        if params.session_id:
            headers['Cookie'] = 'Session_id={0};'.format(params.session_id)
        if params.req_id_header:
            headers[params.req_id_header] = str(uuid.uuid4())
        if params.user_agent:
            headers['User-Agent'] = params.user_agent
        return {
            'allow_redirects': False,
            'data': request_protobuf.SerializeToString(),
            'headers': headers,
            'timeout': params.request_timeout
        }


class RequestsRpcClient(iface.IHttpRpcClient):
    """
    Implementation of HTTP RPC client, which uses requests library to perform calls.
    """

    RETRY_SLEEP_INTERVALS = range(1, 11)

    def __init__(self, rpc_url, oauth_token=None, session_id=None, request_timeout=10, sleep_func=time.sleep):
        self._rpc_url = None
        self.rpc_url = rpc_url
        params = RequestParams()
        params.oauth_token = oauth_token
        params.session_id = session_id
        params.request_timeout = request_timeout
        self._params_processor = RequestParamsProcessor(params)
        self._request_params = params
        self._sleep_func = sleep_func

    # ########### Getters/setters part
    @property
    def rpc_url(self):
        return self._rpc_url

    @rpc_url.setter
    def rpc_url(self, value):
        if not value.startswith('http://') and not value.startswith('https://'):
            raise ValueError('RPC URL must start with http(s): {0}'.format(value))
        self._rpc_url = value.rstrip('/')

    # ########### Interface implementation
    def call_remote_method(self, method_name, request_protobuf, response_protobuf, **kwargs):
        url = '{0}/{1}/'.format(self.rpc_url, method_name)
        # Prepare request params if something is overridden
        kwargs = self._params_processor.prepare_request(request_protobuf, kwargs)
        for i in self.RETRY_SLEEP_INTERVALS:
            response = requests.post(url, **kwargs)
            if response.status_code == 429:  # TooManyRequestsError, retry
                self._sleep_func(i)
            elif response.status_code == 200:  # Response is ok, deserialize
                response_protobuf.MergeFromString(response.content)
                return
            else:
                break
        # Error happened, read from body and raise exception
        raise make_exception_from_response(response)


class SessionedRpcClient(iface.IHttpRpcClient):
    """
    Implementation of HTTP RPC client, which uses requests library to perform calls
    and tries to reuse connection using requests.Session().

    Don't forget to call .close() to close connections after use.

    Warning: please use with caution as sessions may rot if not used for some time and subsequent
    requests will fail with connection errors.
    """
    RETRY_SLEEP_INTERVALS = range(1, 11)

    def __init__(self, rpc_url, oauth_token=None, session_id=None, request_timeout=10, sleep_func=time.sleep):
        self._rpc_url = None
        self.rpc_url = rpc_url
        params = RequestParams()
        params.oauth_token = oauth_token
        params.session_id = session_id
        params.request_timeout = request_timeout
        self._params_processor = RequestParamsProcessor(params)
        self._session = requests.Session()
        self._sleep_func = sleep_func

    def close(self):
        self._session.close()

    # ########### Getters/setters part
    @property
    def rpc_url(self):
        return self._rpc_url

    @rpc_url.setter
    def rpc_url(self, value):
        if not value.startswith('http://') and not value.startswith('https://'):
            raise ValueError('RPC URL must start with http(s): {0}'.format(value))
        self._rpc_url = value.rstrip('/')

    # ########### Interface implementation
    def call_remote_method(self, method_name, request_protobuf, response_protobuf, **kwargs):
        url = '{0}/{1}/'.format(self.rpc_url, method_name)
        # Prepare request params if something is overridden
        kwargs = self._params_processor.prepare_request(request_protobuf, kwargs)
        for i in self.RETRY_SLEEP_INTERVALS:
            response = self._session.post(url, **kwargs)
            if response.status_code == 429:  # TooManyRequestsError, retry
                self._sleep_func(i)
            elif response.status_code == 200:  # Response is ok, deserialize
                response_protobuf.MergeFromString(response.content)
                return
            else:
                break
        # Error happened, read from body and raise exception
        raise make_exception_from_response(response)


class RetryingRpcClient(iface.IHttpRpcClient):

    DEFAULT_RETRY_SLEEPER = retry_sleeper.RetrySleeper(max_tries=5, delay=1)
    PERMITTED_SCHEMAS = (
        'http://',
        'https://',
        'http+unix://',
    )

    def __init__(self, rpc_url, oauth_token=None, session_id=None, request_timeout=10, retry_sleeper=None,
                 retry_429=True, retry_5xx=False, retry_connection_errors=False, adapters=None, req_id_header=None,
                 exception_parser=None, path_trailing_slash=True, user_agent=None):
        """
        :type rpc_url: unicode
        :type oauth_token: unicode
        :type session_id: unicode
        :type request_timeout: int | float
        :type retry_sleeper: iface.IRetrySleeper
        :type adapters: dict | None
        :type req_id_header: unicode | None
        :type exception_parser: collections.Callable
        :type path_trailing_slash: bool
        :type user_agent: str
        """
        self._rpc_url = None
        self.rpc_url = rpc_url
        params = RequestParams()
        params.oauth_token = oauth_token
        params.session_id = session_id
        params.request_timeout = request_timeout
        params.req_id_header = req_id_header
        params.user_agent = user_agent
        self._params_processor = RequestParamsProcessor(params)
        self._session = requests.Session()
        self._retry_sleeper = retry_sleeper or self.DEFAULT_RETRY_SLEEPER
        self._retry_429 = retry_429
        self._retry_5xx = retry_5xx
        self._retry_connection_errors = retry_connection_errors
        self._mount_adapters(adapters)
        self._exception_parser = exception_parser or make_exception_from_response
        self._path_trailing_slash = path_trailing_slash

    def _mount_adapters(self, adapters):
        """
        :type adapters: dict | None
        """
        if not adapters:
            return
        for scheme, adapter in adapters.iteritems():
            self._session.mount(scheme, adapter)

    def close(self):
        self._session.close()

    # ########### Getters/setters part
    @property
    def rpc_url(self):
        return self._rpc_url

    @rpc_url.setter
    def rpc_url(self, value):
        found = False
        for s in self.PERMITTED_SCHEMAS:
            if value.startswith(s):
                found = True
                break
        if not found:
            raise ValueError('RPC URL must start with http(s)/http+unix: {0}'.format(value))
        self._rpc_url = value.rstrip('/')

    def _is_retry_needed(self, resp):
        if self._retry_429 and resp.status_code == requests.codes.TOO_MANY_REQUESTS:
            return True
        if self._retry_5xx and 500 <= resp.status_code < 600:
            return True
        return False

    # ########### Interface implementation
    def call_remote_method(self, method_name, request_protobuf, response_protobuf, **kwargs):
        if self._path_trailing_slash:
            url = '{0}/{1}/'.format(self.rpc_url, method_name)
        else:
            url = '{0}/{1}'.format(self.rpc_url, method_name)
        # Prepare request params if something is overridden
        kwargs = self._params_processor.prepare_request(request_protobuf, kwargs)
        sleeper = self._retry_sleeper.copy()
        response = None
        while True:
            try:
                response = self._session.post(url, **kwargs)
            except requests.ConnectionError:
                if not self._retry_connection_errors:
                    raise
                if not sleeper.increment(exception=False):
                    raise
            else:
                if response.status_code == 200:  # Response is ok, deserialize
                    response_protobuf.MergeFromString(response.content)
                    return
                if not self._is_retry_needed(response):
                    break
                if not sleeper.increment(exception=False):
                    break
        # Error happened, read from body and raise exception
        raise self._exception_parser(response)
