from __future__ import unicode_literals

import time
import random

from . import iface
from . import exceptions


class RetrySleeper(iface.IRetrySleeper):
    """
    A retry sleeper that will track its jitter, backoff and
    sleep appropriately when asked.
    """

    def __init__(self, max_tries=-1, delay=0.5, backoff=2, max_jitter=0.8,
                 max_delay=600, sleep_func=time.sleep):
        """
        Create a :class:`RetrySleeper` instance

        :param max_tries: How many times to retry the command. -1 for unlimited.
        :param delay: Initial delay between retry attempts.
        :param backoff: Backoff multiplier between retry attempts.
                        Defaults to 2 for exponential backoff.
        :param max_jitter: Additional max jitter period to wait between
                           retry attempts to avoid slamming the server.
        :param max_delay: Maximum delay in seconds, regardless of other
                          backoff settings. Defaults to ten minutes.
        """
        self.sleep_func = sleep_func
        self.max_tries = max_tries
        self.delay = delay
        self.backoff = backoff
        self.max_jitter = max_jitter
        self.max_delay = float(max_delay)
        self._attempts = 0
        self._cur_delay = delay

    @property
    def cur_delay(self):
        return self._cur_delay

    def reset(self):
        """
        Reset the attempt counter
        """
        self._attempts = 0
        self._cur_delay = self.delay

    def increment(self, exception=True):
        try:
            time_to_sleep = self.get_next_time_to_sleep()
        except exceptions.RpcClientRetryFailedError:
            if exception:
                raise
            return False
        else:
            self.sleep_func(time_to_sleep)
            return True

    def get_next_time_to_sleep(self):
        """
        Increment the failed count and just return delay before the next retry
        and do not sleep
        :rtype: float
        """
        if self._attempts == self.max_tries:
            raise exceptions.RpcClientRetryFailedError("Too many retry attempts")
        self._attempts += 1
        jitter = random.random() * self.max_jitter
        result = min(self._cur_delay + jitter, self.max_delay)
        self._cur_delay = min(self._cur_delay * self.backoff, self.max_delay)
        return result

    def copy(self):
        return RetrySleeper(self.max_tries, self.delay, self.backoff,
                            self.max_jitter, self.max_delay,
                            self.sleep_func)
