from __future__ import unicode_literals

from six.moves import http_client
import json
import uuid

import requests

from sepelib.util.retry import RetrySleeper, Retry

from .errors import ServiceRepoRequestError, ModificationConflictError, NotFoundError


class ServiceRepoClient(object):
    DEFAULT_DELAY = 1.0
    DEFAULT_MAX_DELAY = 5.0
    DEFAULT_ATTEMPTS = 3
    DEFAULT_REQ_TIMEOUT = 30

    RETRY_EXCEPTIONS = (requests.RequestException,)

    def __init__(self, url, token, timeout=None, attempts=None, delay=None, max_delay=None):
        self._base_url = url.rstrip('/')
        self._token = token
        self._timeout = timeout or self.DEFAULT_REQ_TIMEOUT
        self._session = requests.Session()
        self._session.headers['Content-Type'] = 'application/json'
        self._session.headers['Authorization'] = 'OAuth {}'.format(token)
        self._session.headers['Accept-Encoding'] = ''
        self._sleeper = RetrySleeper(
            max_tries=attempts or self.DEFAULT_ATTEMPTS,
            delay=delay or self.DEFAULT_DELAY,
            max_delay=max_delay or self.DEFAULT_MAX_DELAY,
        )

    def get_services(self, category, skip=None):
        """
        :type category: unicode
        :rtype: dict
        """
        params = {'category': category}
        if skip is not None:
            params['skip'] = skip
        return self._request_with_retries(method='GET', url='/v2/services', params=params)

    def get_service(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/'.format(s_id))

    def get_active_runtime_attrs(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/active/runtime_attrs/'.format(s_id))

    def get_history_runtime_attrs(self, s_id, sn_id=''):
        """
        :type s_id: unicode
        :type sn_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/history/runtime_attrs/{}/'.format(s_id, sn_id))

    def get_runtime_attrs(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/runtime_attrs/'.format(s_id))

    def get_current_state(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/current_state/'.format(s_id))

    def get_target_state(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/target_state/'.format(s_id))

    def get_current_state_instances(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/current_state/instances/'.format(s_id))

    def get_instances(self, s_id, sn_id, limit=None, skip=None):
        """
        :type s_id: unicode
        :type sn_id: unicode
        :type limit: int
        :type skip: int
        :rtype: dict
        """
        params = {'limit': limit, 'skip': skip}
        return self._request_with_retries('GET', '/v2/services/instances/{}/sn/{}/'.format(s_id, sn_id), params=params)

    def put_runtime_attrs(self, s_id, content):
        """
        :type s_id: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services/{}/runtime_attrs/'.format(s_id)
        return self._request_with_retries('PUT', url, data=content)

    def get_info_attrs(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/info_attrs/'.format(s_id))

    def put_info_attrs(self, s_id, content):
        """
        :type s_id: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services/{}/info_attrs/'.format(s_id)
        return self._request_with_retries('PUT', url, data=content)

    def get_auth_attrs(self, s_id):
        """
        :type s_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services/{}/auth_attrs/'.format(s_id))

    def put_auth_attrs(self, s_id, content):
        """
        :type s_id: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services/{}/auth_attrs/'.format(s_id)
        return self._request_with_retries('PUT', url, data=content)

    def update_service_resources(self, s_id, resources_data):
        """
        :type s_id: unicode
        :type resources_data: dict
        :rtype: dict
        """
        url = '/v2/services/{}/runtime_attrs/resources/'.format(s_id)
        return self._request_with_retries('PUT', url, data=resources_data)

    def set_service_sandbox_files(self, s_id, resources, comment=''):
        """
        :param resources: a list of dictionaries describing sandbox resources.
        Example:
        >>> [
        >>>     {
        >>>         u'is_dynamic': False,
        >>>         u'local_path': u'bundle.tar',
        >>>         u'resource_type': u'PYTHON_BUNDLE',
        >>>         u'task_id': u'142228712',
        >>>         u'task_type': u'BUILD_YDL'
        >>>     },
        >>>     ...
        >>> ]

        :returns: new runtime attributes

        :type s_id: str | unicode
        :type resources: Iterable[dict]
        :type comment: str | unicode
        :rtype: dict
        """
        runtime_attrs = self.get_runtime_attrs(s_id)
        runtime_attrs['snapshot_id'] = runtime_attrs.pop('_id')
        runtime_attrs['comment'] = comment

        if 'ticket_info' not in runtime_attrs['meta_info']:
            runtime_attrs['meta_info']['ticket_info'] = {}

        runtime_attrs['meta_info']['ticket_info']['release_id'] = ''
        runtime_attrs['meta_info']['ticket_info']['ticket_id'] = ''

        runtime_attrs['content']['resources']['sandbox_files'] = resources
        return self.put_runtime_attrs(s_id, runtime_attrs)

    def create_event(self, s_id, content):
        """
        :type s_id: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services/{}/events/'.format(s_id)
        return self._request_with_retries('POST', url, data=content)

    def create_service(self, content):
        """
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services/'
        return self._request_with_retries('POST', url, data=content)

    def delete_service(self, s_id):
        """
        :type s_id: unicode
        """
        url = '/v2/services/{}/'.format(s_id)
        return self._request_with_retries('DELETE', url)

    def list_dashboards(self):
        """
        :rtype: list
        """
        return self._request_with_retries('GET', '/v2/services_dashboards/catalog/')['result']

    def get_dashboard(self, dashboard_id):
        """
        :type dashboard_id: unicode
        :rtype: dict
        """
        return self._request_with_retries('GET', '/v2/services_dashboards/catalog/{}/'.format(dashboard_id))

    def create_dashboard(self, content):
        """
        :type content: dict
        :rtype: dict
        """
        return self._request_with_retries('POST', '/v2/services_dashboards/catalog/', data=content)

    def put_dashboard(self, dashboard_id, content):
        """
        type dashboard_id: unicode
        type content: dict
        rtype: dict
        """
        url = '/v2/services_dashboards/catalog/{}/'.format(dashboard_id)
        return self._request_with_retries('PUT', url, data=content)

    def list_dashboard_recipes(self, dashboard_id):
        """
        type dashboard_id: unicode
        rtype: dict
        """
        url = '/v2/services_dashboards/catalog/{}/recipes/'.format(dashboard_id)
        return self._request_with_retries('GET', url)

    def get_dashboard_recipe(self, dashboard_id, recipe_id):
        """
        :type dashboard_id: unicode
        :type recipe_id: unicode
        :rtype: dict
        """
        url = '/v2/services_dashboards/catalog/{}/recipes/{}/'.format(dashboard_id, recipe_id)
        return self._request_with_retries('GET', url)

    def create_dashboard_recipe(self, dashboard_id, content):
        """
        :type dashboard_id: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services_dashboards/catalog/{}/recipes/'.format(dashboard_id)
        return self._request_with_retries('POST', url, data=content)

    def update_dashboard_recipe(self, dashboard_id, recipe_id, content):
        """
        :type dashboard_id: unicode
        :type recipe_id: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services_dashboards/catalog/{}/recipes/{}/'.format(dashboard_id, recipe_id)
        return self._request_with_retries('PUT', url, data=content)

    def set_instance_target_state(self, s_id, sn_id, engine, slot, content):
        """
        :type s_id: unicode
        :type sn_id: unicode
        :type engine: unicode
        :type slot: unicode
        :type content: dict
        :rtype: dict
        """
        url = '/v2/services/instances/{}/sn/{}/engines/{}/slots/{}/set_target_state/'.format(s_id, sn_id, engine, slot)
        return self._request_with_retries('POST', url, data=content)

    def _request_with_retries(self, method, url, params=None, data=None):
        retry = Retry(retry_sleeper=self._sleeper.copy(), retry_exceptions=self.RETRY_EXCEPTIONS)
        try:
            return retry(self._request, method, url, params=params, data=data)
        except requests.HTTPError as e:
            msg = 'Request "{}" failed with {} {}: "{}"'.format(e.request.url,
                                                                e.response.status_code,
                                                                e.response.reason,
                                                                e.response.content)
            raise ServiceRepoRequestError(msg)
        except requests.RequestException as e:
            raise ServiceRepoRequestError('Service repo client request error: {}'.format(e))

    def _request(self, method, relative_url, params=None, data=None):
        url = '{}/{}/'.format(self._base_url, relative_url.lstrip('/').rstrip('/'))

        if isinstance(data, dict):
            data = json.dumps(data)

        response = self._session.request(
            method, url=url, params=params, data=data, timeout=self._timeout, headers={'X-Req-Id': uuid.uuid4().hex}
        )

        if response.status_code == http_client.NOT_FOUND:
            raise NotFoundError(response=response)
        elif response.status_code == http_client.CONFLICT:
            raise ModificationConflictError(response=response)
        elif http_client.BAD_REQUEST <= response.status_code < http_client.INTERNAL_SERVER_ERROR:
            msg = 'Service repo client request error: {} {}'.format(response.status_code, response.text)
            raise ServiceRepoRequestError(msg)
        response.raise_for_status()
        if method in ('POST', 'PUT', 'GET'):
            return response.json()
