from __future__ import unicode_literals
import os
import time
import socket
import argparse
import logging
import sys

from nanny_sox_audit import checker
from nanny_sox_audit import fileutils
from nanny_sox_audit import jugglerutils


def setup_logging():
    logging.basicConfig(stream=sys.stdout,
                        format='[%(asctime)s] %(levelname)s: %(message)s',
                        level=logging.INFO)


def get_checksum(target_file, checksum_file):
    f = os.path.basename(checksum_file)
    if f == 'shard.conf':
        checksum = fileutils.parse_checksum_from_shard_conf(checksum_file)
    elif f == 'dump.json':
        checksum = fileutils.parse_checksum_from_dump_json(checksum_file,
                                                               target_file)
    else:
        raise argparse.ArgumentError('Invalid checksum file')
    return checksum


def main():
    setup_logging()

    parser = argparse.ArgumentParser()
    parser.add_argument('-r', '--root', required=True)
    parser.add_argument('-a', '--archive-file')
    parser.add_argument('-f', '--resource-names', nargs='+')
    parser.add_argument('-c', '--checksum-file', required=True)
    parser.add_argument('-i', '--juggler-instance', default=None)
    parser.add_argument('-d', '--delay', type=int, default=300)
    parser.add_argument('-u', '--juggler-url', default='http://[::1]:31579/events')
    parser.add_argument('--host', default=socket.gethostname())
    parser.add_argument('-s', '--juggler-service', default='nanny_sox_audit')
    args = parser.parse_args()

    checkers = []

    if args.archive_file:
        archive_file_name = os.path.basename(args.archive_file)
        arc_checksum = get_checksum(archive_file_name, args.checksum_file)
        c = checker.ArchiveChecker(args.root, args.archive_file, arc_checksum)
        checkers.append(c)

    if args.resource_names:
        for resource_name in args.resource_names:
            resource_file_name = os.path.basename(resource_name)
            checksum = get_checksum(resource_file_name, args.checksum_file)
            c = checker.ResourceChecker(args.root, resource_name, checksum)
            checkers.append(c)

    while True:
        logging.info('starting check...')
        status = None
        desc = None
        for c in checkers:
            status, desc = c.check()
            if status != 'OK':
                break
        logging.info('check finished: status=%s, description=%s', status, desc)
        r = jugglerutils.push_juggler(url=args.juggler_url,
                                      host=args.host,
                                      service=args.juggler_service,
                                      status=status,
                                      instance=args.juggler_instance,
                                      desc=desc)
        logging.info('pushed to juggler [%s], sleep for %s seconds...', r.status_code, args.delay)
        time.sleep(args.delay)


if __name__ == '__main__':
    main()
