import requests
import click
import yaml

from library.python.vault_client.client import VaultClient
from library.python.vault_client.instances import VAULT_PRODUCTION_API
from library.python.vault_client.errors import ClientError

from library.python import oauth


# configure yaml for python2
yaml.add_representer(unicode, lambda self, data: self.represent_str(data.encode('utf-8')))

GET_YAV_TOKEN_URL = 'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=ce68fbebc76c4ffda974049083729982'
NANNY_ABC_ID = 730
NANNY_URL = 'https://nanny.yandex-team.ru'
NANNY_CLIENT_ID = '256a6d8ce9d846e88035328bb1c857d0'
NANNY_CLIENT_SECRET = 'e63198807b7f4b4d9d786d19fe9cbbf9'


def get_runtime_attrs(session, service_id):
    url = NANNY_URL.rstrip('/') + '/v2/services/{}/runtime_attrs/'.format(service_id)
    r = session.get(url)
    r.raise_for_status()
    content = r.json()
    return {
        'content': content['content'],
        'snapshot_id': content['_id'],
        'comment': ''
    }


def update_runtime_attrs(session, service_id, spec):
    url = NANNY_URL.rstrip('/') + '/v2/services/{}/runtime_attrs/'.format(service_id)
    r = session.put(url, json=spec)
    r.raise_for_status()


def color(status):
    if status in ('OK', 'NEWEST VERSION'):
        return 'green'
    if status in ('NO NEED UPDATE'):
        return 'yellow'
    if status in ('FAILED',):
        return 'red'
    if status in ('NOT FOUND',):
        return None


def echo_status(service_id, status):
    click.secho(status + ' '*(15 - len(status)), fg=color(status), nl=False)
    click.echo(service_id)


@click.command()
@click.option('--id', "secret_id", required=True, help="Secret id", prompt="Enter secret id")
@click.option('--nanny-token')
@click.option('--vault-token')
@click.option('--comment', help="Nanny commit comment")
def cli(secret_id, nanny_token, vault_token, comment):
    if vault_token:
        auth = 'OAuth {}'.format(vault_token)
        vault_client = VaultClient(host=VAULT_PRODUCTION_API, decode_files=True, authorization=auth)
    else:
        vault_client = VaultClient(host=VAULT_PRODUCTION_API, decode_files=True)

    try:
        secret = vault_client.get_secret(secret_id, page_size=1000000000)
    except ClientError as e:
        click.echo("Nanny vault client error: {}".format(e))
        return

    tokens = secret['tokens']
    if not secret['secret_versions']:
        click.echo('Secret has no versions')
        return
    newest_version = secret['secret_versions'][0]['version']
    if not comment:
        comment = 'Bump secret "{}" version'.format(secret['name'])
    services_use_secret = set()
    for token in tokens:
        if token.get('tvm_app', {}).get('abc_department', {}).get('id', 0) == NANNY_ABC_ID:
            services_use_secret.add(token['signature'])
    if not services_use_secret:
        click.echo('No Nanny services use this secret')
        return
    services_use_secret = list(services_use_secret)

    session = requests.session()
    session.headers['Content-Type'] = 'application/json'
    if not nanny_token:
        nanny_token = oauth.get_token(NANNY_CLIENT_ID, NANNY_CLIENT_SECRET)
    session.headers['Authorization'] = 'OAuth {}'.format(nanny_token)

    need_update_services = {}
    for service_id in services_use_secret:
        status = 'UNKNOWN'
        try:
            runtime_attrs = get_runtime_attrs(session, service_id)
        except requests.HTTPError as e:
            if e.response.status_code == 404:
                status = 'NOT FOUND'
            else:
                status = 'FAILED'
                click.echo('Failed on {}'.format(service_id))
                click.echo(e.response.content)
            echo_status(service_id, status)
            continue
        need_update = False
        for container in runtime_attrs['content']['instance_spec'].get('containers', []):
            for env in container.get('env', []):
                value = env['valueFrom']
                if value['type'] != 'VAULT_SECRET_ENV':
                    continue
                env_secret = value['vaultSecretEnv']['vaultSecret']
                if env_secret['secretId'] != secret_id:
                    continue
                if env_secret['secretVer'] == newest_version:
                    continue
                env_secret['secretVer'] = newest_version
                need_update = True
        for volume in runtime_attrs['content']['instance_spec'].get('volume', ''):
            volume_secret = volume['vaultSecretVolume']['vaultSecret']
            if volume_secret['secretId'] != secret_id:
                continue
            if volume_secret['secretVer'] == newest_version:
                continue
            volume_secret['secretVer'] = newest_version
            need_update = True
        if need_update:
            need_update_services[service_id] = runtime_attrs
            status = 'NEED UPDATE'
        else:
            status = "NEWEST VERSION"
        echo_status(service_id, status)

    if not need_update_services:
        click.echo('All founded services have the newest version')
        return

    click.echo()
    click.echo('Do you want to update secret "{}" version in the following services?'.format(secret['name']))
    f = click.prompt('Print yes or no', type=click.Choice(['yes', 'no']))
    if f == 'no':
        return

    for service_id in need_update_services:
        runtime_attrs = need_update_services[service_id]
        runtime_attrs['comment'] = comment
        try:
            update_runtime_attrs(session, service_id, runtime_attrs)
            status = "OK"
        except requests.HTTPError:
            status = "FAILED"
        echo_status(service_id, status)


if __name__ == '__main__':
    cli()
