#!/usr/bin/env python
"""Command-line utility to gather a list of requirements from all setup-*.py
files in this directory.

Useful for setting up a development environment and checking dependencies for
version conflicts.
"""
import tempfile
import glob
import collections
import shutil
import sys
import pkg_resources
import subprocess


def get_requires(setup_py):
    """Returns a list of :class:`pkg_resources.Requirement` of the package
    defined in setup_py.
    """
    tmpdir = tempfile.mkdtemp(prefix='egg-info-tmp-')
    try:
        rv = subprocess.call(['python', setup_py, 'egg_info', '-e', tmpdir],
                             stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        if rv != 0:
            print >> sys.stderr, 'error: egg_info has failed'
            exit(1)
        for dist in pkg_resources.find_distributions(tmpdir, True):
            return dist.requires()
        else:
            return []
    finally:
        shutil.rmtree(tmpdir)


if __name__ == '__main__':
    # maps a :class:`pkg_resources.Requirement` to a list of setup*.py paths
    # that require it.
    setup_py_by_reqs = collections.defaultdict(list)
    # maps a requirement key (:attr:`pkg_resources.Requirement.key`; e.g., "flask")
    # to a list of :class:`Requirement` having this key
    reqs_by_key = collections.defaultdict(list)

    for setup_py in glob.glob('./setup-*.py'):
        package_reqs = get_requires(setup_py)
        for req in package_reqs:
            setup_py_by_reqs[req].append(setup_py)
            reqs_by_key[req.key].append(req)

    for key, reqs in reqs_by_key.iteritems():
        reqs = set(reqs)
        if len(reqs) > 1:
            # having more than one requirement with the same key means a version conflict
            setup_pys = sum((setup_py_by_reqs[req] for req in reqs), [])
            print >> sys.stderr, (
                'error: the following setup-util.py files require different '
                'versions of {0}: {1}'.format(key, ', '.join(setup_pys)))
            exit(1)

        req = str(iter(reqs).next())
        if not req.startswith('sepelib'):
            print req
