"""Service configuration reading/accessing facilities."""

from __future__ import unicode_literals

import codecs
import copy
import os
import re
import six
import jinja2
import yaml

from sepelib.core.exceptions import Error

_CONFIG = None
_UNDEFINED = object()


class MissingConfigOptionError(Error):
    def __init__(self, name):
        super(MissingConfigOptionError, self).__init__("Configuration option '{}' is missing.", name)
        self.name = name


def _parse_key_value(value, pattern=re.compile('^([a-zA-Z_]\w+)=(.*)$')):
    match = pattern.match(value.lstrip())
    if match is None:
        raise ValueError('Provided value {} does not match {}'.format(value, pattern.pattern))
    return match.groups()


def augment_args_parser(args_parser, dest='config_context'):
    """
    Adds config specific argument to pass config values from command line.
    Result is stored in 'config_context' attribute by default.

    :type args_parser: argparse.ArgumentParser
    """
    args_parser.add_argument('-V',
                             action='append',
                             dest=dest,
                             default=[],
                             help="Add key=value variable to config file jinja context.",
                             type=_parse_key_value)
    return args_parser


def get_context_from_env(prefix='SEPELIB', env_getter=lambda: os.environ, ignore_errors=False):
    """
    Extracts config context from environment variables.
    Example:
    env = {
        'PATH': '/bin',
        'SEPELIB_port': '8080',
        'SEPELIB_logpath': '/usr/local/www/logs/sepelib-service.log'
    } =>
    context = {
        'port': '8080',
        'logpath': '/usr/local/www/logs/sepelib-service.log'
    }

    :param prefix: environment variable prefix
    :param env_getter: function to get a dictionary with environment variables
    :param ignore_errors: should we skip key-value parsing errors
    """
    if not isinstance(prefix, six.string_types):
        raise ValueError("Prefix must a string")
    context = {}
    prefix_len = len(prefix) + 1  # SEPELIB + _ (underscore)
    for k, v in six.iteritems(env_getter()):
        if k.startswith(prefix):
            try:
                key, value = _parse_key_value("{}={}".format(k[prefix_len:], v))
                context[key] = value
            except ValueError:
                if not ignore_errors:
                    raise
    return context


def load(path=None, defaults=None, config_context=None):
    """Loads the specified config file."""
    if isinstance(config_context, list):
        # Assume that jinja context is a list of key-value tuples
        config_context = dict(config_context)
    config = {} if path is None else _load(path, jinja_context=config_context)
    if defaults is not None:
        _merge(config, _load(defaults, jinja_context=config_context))

    global _CONFIG
    _CONFIG = config

    return config


def get():
    """Returns loaded configuration."""

    if _CONFIG is None:
        raise Error("Configuration file is not loaded yet.")

    return _CONFIG


def merge(config):
    """Merges loaded config with the specified one."""

    _merge(get(), config)


def get_value(name, default=_UNDEFINED, config=None):
    """Returns the specified configuration value."""

    value = get() if config is None else config

    for key in name.split("."):
        if not isinstance(value, dict):
            raise MissingConfigOptionError(name)

        try:
            value = value[key]
        except KeyError:
            if default is _UNDEFINED:
                raise MissingConfigOptionError(name)

            return default

    return value


def set_value(name, value, config=None):
    """Sets the specified configuration value."""

    if config is None:
        config = get()

    keys = name.split(".")

    for key_id, key in enumerate(keys):
        if not isinstance(config, dict):
            raise Error("Unable to set configuration value '{}': '{}' is not a dictionary.",
                        name, ".".join(keys[:key_id]) or "[root]")

        if key_id == len(keys) - 1:
            config[key] = value
        else:
            config = config.setdefault(key, {})


def _load(path, jinja_context=None):
    try:
        with codecs.open(path, encoding="utf-8") as config_file:
            file_content = config_file.read()
    except Exception as e:
        raise Error("Error while reading configuration file '{}': {}.", path, e)
    if jinja_context is not None:
        try:
            file_content = jinja2.Template(file_content).render(**jinja_context)
        except Exception as e:
            raise Error("Error while applying context to configuration file '{}': {}.", path, e)
    try:
        return yaml.load(file_content, Loader=yaml.SafeLoader)
    except Exception as e:
        raise Error("Error while parsing configuration file '{}': {}.", path, e)


def _merge(config, other):
    for key, value in six.iteritems(other):
        if key in config:
            if isinstance(config[key], dict) and isinstance(value, dict):
                _merge(config[key], value)
        else:
            config[copy.deepcopy(key)] = copy.deepcopy(value)
