import calendar

from flask import request, make_response, abort

from .mime import mime_render, MimeException
from .util import date_to_str


__all__ = ['prep_not_modified', 'err_response', 'prep_response']


def prep_not_modified(last_modified):
    if_modified_since = request.if_modified_since
    if if_modified_since:
        # if_modified_since.timetuple() returns tuple in UTC
        # and e.g. time.mktime assumes that this tuple in local time
        # so we use calendar module here
        if_modified_since = calendar.timegm(if_modified_since.timetuple())
        # if_modified_since looses milliseconds
        if int(last_modified) <= if_modified_since:
            return make_response('', 304)
    return None


def err_response(type_, message, code=500, fmt=None):
    document = {
        'error': type_,
        'message': message
    }
    try:
        body, mimetype = mime_render(document, fmt)
    except MimeException as e:
        return make_response(e.reason, 400)
    response = make_response(body, code)
    response.mimetype = mimetype
    return response


def prep_response(document, last_modified=None, fmt=None, code=200, headers=None):
    """
    Create ReST response, formatting document and adding headers if needed.
    :param document: object
    :param last_modified: float or None
    :param fmt: str or None
    """
    if document is not None:
        resp = prep_not_modified(last_modified)
        if resp is not None:
            return resp

        try:
            body, mimetype = mime_render(document, fmt)
        except MimeException as e:
            abort(400, description=e.reason)
        response = make_response(body, code)
        response.mimetype = mimetype
        if last_modified:
            response.headers.add('Last-Modified', date_to_str(last_modified))
        if headers:
            response.headers.extend(headers)
        return response
    return err_response("not_found", "object not found", code=404)


_EMPTY_DOC = {}


def empty_doc_response(document=_EMPTY_DOC, code=200, fmt=None, headers=None):
    """
    Shortcut for `prep_response` to return empty document (not empty body).
    """
    return prep_response(document, fmt=fmt, code=code, headers=headers)
