import functools
import logging
import time
import traceback

from flask import request, g, current_app
from flask import render_template_string, jsonify

import six
from pymongo import errors as pymongo_errors
from sepelib.util.exc import format_exc
try:
    from kazoo.exceptions import KazooException
except ImportError:
    KazooException = None
from sepelib.yandex.blackbox import BlackboxError
from .mime import request_wants_json

__all__ = ['exception_handler']

log = logging.getLogger('flask-error-handlers')


class TokenBucket(object):
    """An implementation of the token bucket algorithm.

    >>> bucket = TokenBucket(80, 0.5)
    >>> print bucket.consume(10)
    True
    >>> print bucket.consume(90)
    False
    """
    def __init__(self, tokens, fill_rate, time=time.time):
        """tokens is the total tokens in the bucket. fill_rate is the
        rate in tokens/second that the bucket will be refilled."""
        self.capacity = float(tokens)
        self._tokens = float(tokens)
        self.fill_rate = float(fill_rate)
        self.time = time
        self.timestamp = time()

    def consume(self, tokens):
        """Consume tokens from the bucket. Returns True if there were
        sufficient tokens otherwise False."""
        if tokens <= self.tokens:
            self._tokens -= tokens
        else:
            return False
        return True

    def get_tokens(self):
        if self._tokens < self.capacity:
            now = self.time()
            delta = self.fill_rate * (now - self.timestamp)
            self._tokens = min(self.capacity, self._tokens + delta)
            self.timestamp = now
        return self._tokens
    tokens = property(get_tokens)

# Rate limit on logging exception (log full stack trace) no more than this rate.
exc_logging_tokens = TokenBucket(tokens=1, fill_rate=1)  # Once per seconds.


_ERROR_TEMPLATE = """
<html>
<head>
    <meta charset="utf-8">
    <style>
    .alert {
        padding: 8px 35px 8px 14px;
        margin-bottom: 20px;
        text-shadow: 0 1px 0 rgba(255, 255, 255, 0.5);
        background-color: #fcf8e3;
        border: 1px solid #fbeed5;
        -webkit-border-radius: 4px;
            -moz-border-radius: 4px;
                border-radius: 4px;
    }
    .alert-error {
        color: #b94a48;
        background-color: #f2dede;
        border-color: #eed3d7;
    }
    strong {
        font-weight: bold;
    }
    pre {
      padding: 0 3px 2px;
      font-family: Monaco, Menlo, Consolas, "Courier New", monospace;
      font-size: 12px;
      color: #333333;
      -webkit-border-radius: 3px;
         -moz-border-radius: 3px;
              border-radius: 3px;
      display: block;
      padding: 9.5px;
      margin: 0 0 10px;
      font-size: 13px;
      line-height: 20px;
      word-break: break-all;
      word-wrap: break-word;
      white-space: pre;
      white-space: pre-wrap;
      background-color: #f5f5f5;
      border: 1px solid #ccc;
      border: 1px solid rgba(0, 0, 0, 0.15);
      -webkit-border-radius: 4px;
         -moz-border-radius: 4px;
              border-radius: 4px;
    </style>
    <title>Error happened</title>
</head>
<div class="container">
    <div class="alert alert-error">
        <strong>Oh, snap!</strong> {{ reason or message }}
        {% if traceback %}
            <pre>{{ traceback }}</pre>
        {% endif %}
    </div>
</div>
</html>
"""


def exception_handler(f):
    """
    Call exception handler, which should return error message
    and http error code. If current request is AJAX - return json,
    otherwise - return html page.
    """
    @functools.wraps(f)
    def wrapper(e, *args, **kwargs):
        if 'sentry' in current_app.extensions:
            current_app.extensions['sentry'].captureException()

        result = f(e, *args, **kwargs)
        message, status, headers = result + (tuple() if len(result) > 2 else ({},))

        if request_wants_json() or request.is_xhr:
            response = jsonify({
                "result": 'FAIL',
                "message": message,
                "errors": []
            })
            response.headers.extend(headers)
            response.status_code = status
            return response
        else:
            return render_template_string(_ERROR_TEMPLATE,
                                          message=message,
                                          traceback=traceback.format_exc()), status, headers

    return wrapper


# ==== exception handlers ====
@exception_handler
def not_authorized(_):
    login = g.identity.id if hasattr(g, 'identity') else "<unknown>"
    message = u"User '{0}' is not authorized.".format(login)
    return message, 403


@exception_handler
def mongo_error(e):
    message = format_exc('MongoDb error', e)
    log.error(message)
    return message, 500


@exception_handler
def coord_error(e):
    # NoNodeError, SessionExpiredError do not have any args, so let's print class
    # to be more meaningful
    message = "{0} error: {1}".format(e.__class__.__name__, six.text_type(e).encode('utf-8'))
    log.error(message)
    return message, 500


@exception_handler
def error_happened(e):
    message = format_exc('Unexpected error', e)
    # we should log traceback ourselves
    # exc_info is preserved here, so we can use standard procedure
    if exc_logging_tokens.consume(1):
        log.exception(message)
    else:
        log.error(message)
    return message, 500


@exception_handler
def blackbox_error(e):
    message = format_exc('Blackbox request failed', e)
    log.error(message)
    return message, 500


def register_error_handlers(app):
    """
    Register common error handlers for :param app:
    """
    # beware the order
    # broader cases will match first
    # like in ordinary try-except clause
    app.errorhandler(403)(not_authorized)
    if not app.testing:
        app.errorhandler(pymongo_errors.PyMongoError)(mongo_error)
        if KazooException is not None:
            app.errorhandler(KazooException)(coord_error)
        app.errorhandler(BlackboxError)(blackbox_error)
        app.errorhandler(Exception)(error_happened)
