# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import os
import mmap
import tempfile
import errno
import fcntl
import shutil


def set_close_exec(fd):
    """
    Helper to add CLOEXEC to provided file descriptor.
    :type fd: int
    """
    flags = fcntl.fcntl(fd, fcntl.F_GETFD)
    fcntl.fcntl(fd, fcntl.F_SETFD, flags | fcntl.FD_CLOEXEC)


def close_ignore(fd, ignore=(errno.EBADF, errno.EINTR)):
    """
    Close :param fd: ignoring errors :param ignore:
    """
    while 1:
        try:
            os.close(fd)
        except EnvironmentError as e:
            if e.errno in ignore:
                break


def get_temp_dir(prefix=None, place=None, chmod=None):
    """
    Получить временную директорию для данных, настроить её права

    :param prefix: префикс в названии директории
    :param place: где создавать директорию
    :param chmod: какие права установить на созданную директорию
    :return: путь до созданной директории
    :rtype: str
    """
    if prefix is not None:
        tmp_directory_path = tempfile.mkdtemp(prefix=prefix, dir=place)
    else:
        tmp_directory_path = tempfile.mkdtemp(dir=place)
    if chmod:
        os.chmod(tmp_directory_path, chmod)
    return tmp_directory_path


class TempDir(object):
    """
    Создание временной папки и удаление её, если это нужно
    """
    def __init__(self, prefix=None, directory=None, remove=True):
        """
        :param prefix: префикс для создаваемой папки
        :param directory: где создавать папку, по умолчанию в системной временной директории
        :param remove: выполнять ли удаление папки при выходе
        :type prefix: str | unicode
        :type directory: str | unicode
        :type remove: bool
        """
        self.created_directory = None
        self.parent_directory = directory
        self.prefix = prefix
        self.remove = remove

    def __enter__(self):
        if self.parent_directory:
            makedirs_ignore(self.parent_directory)

        self.created_directory = get_temp_dir(
            prefix=self.prefix, place=self.parent_directory
        )
        return self.created_directory

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.created_directory and self.remove:
            try:
                shutil.rmtree(self.created_directory)
            except OSError as err:
                if err.errno != errno.ENOENT:
                    raise


def get_last_n_symbols_from_file(file_path, symbols_number=500):
    """
        Вернуть последние N символов из файла

        :param file_path: путь до файла
        :param symbols_number: сколько символов считать
        :result: последнеие N символов или None, если не получилось считать
        :rtype: str or None
    """
    try:
        with open(file_path) as f:
            # идём в конец файла
            f.seek(0, os.SEEK_END)
            # смотрим сколько в нём байт
            file_size = f.tell()
            # считываем или с конца файла, или весь файл, если он маленький
            if symbols_number < file_size:
                f.seek(-symbols_number, os.SEEK_END)
                result = f.read(symbols_number)
            else:
                f.seek(0, os.SEEK_SET)
                result = f.read(symbols_number)
    except Exception:
        return None
    return result


def lock_fd(fd):
    """
    Lock a file with its file descriptor.

    :param fd: file descriptor
    :return: True is the file was locked; False otherwise.
    :rtype: bool
    """
    try:
        fcntl.lockf(fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except (IOError, OSError):
        return False
    return True


def unlock_fd(fd):
    """
    Unlock a file with its file descriptor.

    :param fd: file descriptor
    :return: True is the file was unlocked; False otherwise.
    :rtype: bool
    """
    try:
        fcntl.lockf(fd, fcntl.LOCK_UN)
    except (IOError, OSError):
        return False
    return True


def try_lock_fd(fd):
    """
    Try to lock a file by its descriptor.

    :param fd: file descriptor
    :return: True if we locked the file successfully; False otherwise.
    :rtype: bool
    """
    try:
        result = lock_fd(fd)
        if result:
            unlock_fd(fd)
        return result
    except IOError:
        return True


def check_file_locked(file_path):
    """
    Check a file path - is it locked or not.

    :param basestring file_path: path to a file
    :return: True is the path is locked; False otherwise.
    :rtype: bool
    """
    try:
        with open(file_path, 'a') as file_lock_fd:
            if try_lock_fd(file_lock_fd):
                return False
        return True
    except (IOError, OSError):
        return False


def makedirs_ignore(dir_path, mode=0o777):
    """
    Recursively creates directory :param dir_path: and all intermediate-level dirs ignoring EEXIST error

    :type dir_path: str | unicode
    :type mode: int
    """
    try:
        os.makedirs(dir_path, mode=mode)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise


def atomic_write(file_path, contents, mode=0o664):
    """
    Writes :param contents: to :param file_path: via renaming temporary file

    Created file will have permissions specified in :param mode:
    """
    if isinstance(contents, unicode):
        contents = contents.encode('utf-8')

    with tempfile.NamedTemporaryFile(dir=os.path.dirname(file_path), delete=False) as fd:
        fd.write(contents)
        fd.flush()
        os.fdatasync(fd.fileno())

    os.chmod(fd.name, mode)
    os.rename(fd.name, file_path)


def remove_ignore(filename):
    """
    Removes :param filename: ignoring ENOENT error
    """
    try:
        os.remove(filename)
    except OSError as e:
        if e.errno != errno.ENOENT:
            raise


class FileLinesChunk(object):
    """
    Stores lines read from file.
    """

    __slots__ = ['content', 'start_offset']

    def __init__(self, content, start_offset):
        self.content = content
        self.start_offset = start_offset

    def __repr__(self):
        return 'FileLinesChunk(content={!r}, offset={!r})'.format(self.content, self.offset)

    __str__ = __repr__


def read_file_last_lines(filename, lines_to_read, offset=None, ignore_last_line_end=True, encoding='utf8',
                         line_end=b'\n', byte_limit=None):
    """
    Reads last :param lines_to_read: lines from file :param filename:.

    If :param offset: is int, lines will be read back from the given byte offset from
    the beginning of the file. If :param offset: is None, lines will be read from the
    end of the file. Not more than :param byte_limit: bytes will be read in total.

    :type filename: unicode
    :type offset: int|NoneType
    :type lines_to_read: int
    :type encoding: unicode
    :type line_end: str
    :type ignore_last_line_end: bool

    :rtype: FileLinesChunk
    """
    result = b''
    lines_read = 0
    bytes_to_read = mmap.PAGESIZE

    with open(filename) as fd:

        if offset is not None:
            fd.seek(offset, os.SEEK_SET)
        else:
            fd.seek(0, os.SEEK_END)
            offset = fd.tell()

        while offset > 0 and lines_read < lines_to_read:
            offset -= bytes_to_read

            if offset < 0:
                bytes_to_read += offset
                offset = 0

            fd.seek(offset)
            read = fd.read(bytes_to_read)
            lines_read += read.count(line_end)
            result = read + result

            if byte_limit is not None and len(result) > byte_limit:
                result = result[-byte_limit:]
                break

            if ignore_last_line_end and result.endswith(line_end):
                lines_to_read += 1
                ignore_last_line_end = False

    start_pos = 0
    while lines_read >= lines_to_read:
        start_pos = result.find(line_end, start_pos) + 1
        lines_read -= 1

    return FileLinesChunk(content=result[start_pos:].decode(encoding),
                          start_offset=offset + start_pos)
