"""
Copy-paste from kazoo eliminating some kazoo specific logic.
"""
import logging
import random

from gevent import sleep, Timeout


log = logging.getLogger(__name__)


class ForceRetryError(Exception):
    """
    Raised when some logic wants to force a retry
    """
    pass


class RetryFailedError(Exception):
    """
    Raised when retrying an operation ultimately failed, after
    retrying the maximum number of attempts.
    """
    pass


class RetrySleeper(object):
    """
    A retry sleeper that will track its jitter, backoff and
    sleep appropriately when asked.
    """
    def __init__(self, max_tries=-1, delay=0.5, backoff=2, max_jitter=0.8,
                 max_delay=600, sleep_func=sleep):
        """
        Create a :class:`RetrySleeper` instance

        :param max_tries: How many times to retry the command. -1 for unlimited.
        :param delay: Initial delay between retry attempts.
        :param backoff: Backoff multiplier between retry attempts.
                        Defaults to 2 for exponential backoff.
        :param max_jitter: Additional max jitter period to wait between
                           retry attempts to avoid slamming the server.
        :param max_delay: Maximum delay in seconds, regardless of other
                          backoff settings. Defaults to ten minutes.

        """
        self.sleep_func = sleep_func
        self.max_tries = max_tries
        self.delay = delay
        self.backoff = backoff
        self.max_jitter = max_jitter
        self.max_delay = float(max_delay)
        self._attempts = 0
        self._cur_delay = delay

    @property
    def cur_delay(self):
        return self._cur_delay

    def reset(self):
        """
        Reset the attempt counter
        """
        self._attempts = 0
        self._cur_delay = self.delay

    def increment(self, exception=True):
        """
        Increment the failed count, and sleep appropriately before
        continuing
        :return: False if max attempts reached, True otherwise
        :rtype: bool
        """
        try:
            time_to_sleep = self.get_next_time_to_sleep()
        except RetryFailedError:
            if exception:
                raise
            return False
        else:
            self.sleep_func(time_to_sleep)
            return True

    def get_next_time_to_sleep(self):
        """
        Increment the failed count and just return delay before the next retry
        and do not sleep
        :rtype: float
        """
        if self._attempts == self.max_tries:
            raise RetryFailedError("Too many retry attempts")
        self._attempts += 1
        jitter = random.random() * self.max_jitter
        result = min(self._cur_delay + jitter, self.max_delay)
        self._cur_delay = min(self._cur_delay * self.backoff, self.max_delay)
        return result

    def copy(self):
        """
        Return a clone of this retry sleeper
        """
        return RetrySleeper(self.max_tries, self.delay, self.backoff,
                            self.max_jitter, self.max_delay,
                            self.sleep_func)


class Retry(object):
    """
    Helper for retrying a method in the face of retry-able
    exceptions
    """

    def __init__(self, retry_sleeper, retry_exceptions=(Exception,)):
        """
        Create a :class:`Retry` instance for retrying function
        calls

        :param retry_sleeper: :class:`RetrySleeper` to be used
        :param retry_exceptions: tuple of exceptions to be retried
        """
        self.retry_sleeper = retry_sleeper
        self.retry_exceptions = retry_exceptions + (ForceRetryError,)
        self.last_error = None

    def __call__(self, func, *args, **kwargs):
        """
        Call a function with arguments until it completes without
        throwing an exception

        :param func: Function to call
        :param args: Positional arguments to call the function with
        :param kwargs: Keyword arguments to call the function with

        The function will be called until it doesn't throw one of the
        retryable exceptions.
        """
        self.retry_sleeper.reset()

        while True:
            try:
                return func(*args, **kwargs)
            except self.retry_exceptions:
                if not self.retry_sleeper.increment(exception=False):
                    raise


class RetryWithTimeout(Retry):
    """
    Retry with every called wrapped with timeout.
    """
    def __init__(self, attempt_timeout, retry_sleeper, retry_exceptions=(Exception, Timeout)):
        super(RetryWithTimeout, self).__init__(retry_sleeper, retry_exceptions)
        self._attempt_timeout = attempt_timeout

    def __call__(self, func, *args, **kwargs):
        def call_with_timeout():
            with Timeout(self._attempt_timeout):
                return func(*args, **kwargs)
        return super(RetryWithTimeout, self).__call__(call_with_timeout)
