from __future__ import unicode_literals

from nanny_repo import repo_pb2

from nanny_repo import repo_api_pb2
from nanny_rpc_client import exceptions as rpc_exceptions

from service_repo_client.lib import pbutil
from service_repo_client.lib import template_env
from service_repo_client.lib.resource_manager import ResourceManager


class IAspectCtrl(object):
    aspect_name = NotImplemented
    template_name = NotImplemented
    rendered_name = NotImplemented
    raw_content_name = NotImplemented

    def get_attrs(self, s_id):
        raise NotImplementedError

    def get_version(self, content):
        raise NotImplementedError

    def put_attrs(self, s_id, content, comment):
        raise NotImplementedError

    def convert_to_yml(self, service_path, content):
        raise NotImplementedError

    def convert_to_json(self, service_path, content):
        raise NotImplementedError

    def get_template_env(self, s_id, service_path):
        raise NotImplementedError


class CleanupPolicyCtrl(IAspectCtrl):
    aspect_name = 'cleanup_policy'
    template_name = 'template_cleanup_policy.yml'
    rendered_name = 'cleanup_policy.yml'
    raw_content_name = 'cleanup_policy.json'

    def __init__(self, repo_stub):
        self.repo_stub = repo_stub

    def get_attrs(self, s_id):
        req = repo_api_pb2.GetCleanupPolicyRequest(policy_id=s_id)
        try:
            resp = self.repo_stub.get_cleanup_policy(req)
        except rpc_exceptions.NotFoundError:
            return None
        else:
            return pbutil.pb_to_jsondict(resp)

    def put_attrs(self, s_id, content, comment):
        meta = repo_pb2.CleanupPolicyMeta()
        spec = repo_pb2.CleanupPolicySpec()
        pbutil.jsondict_to_pb(content['policy']['meta'], meta)
        pbutil.jsondict_to_pb(content['policy']['spec'], spec)
        req = repo_api_pb2.UpdateCleanupPolicyRequest(meta=meta, spec=spec)
        resp = self.repo_stub.update_cleanup_policy(req)
        return pbutil.pb_to_jsondict(resp)

    def get_version(self, content):
        return content['policy']['meta']['version'] if content else None

    def convert_to_json(self, service_path, content):
        return content

    def convert_to_yml(self, service_path, content):
        return content

    def get_template_env(self, s_id, service_path):
        return template_env.TemplateEnv(s_id, service_path).get_env()


class AttrsCtrl(IAspectCtrl):
    def __init__(self, rest_client):
        self.client = rest_client

    def get_version(self, content):
        return content['_id'] if content else None

    def get_template_env(self, s_id, service_path):
        return template_env.TemplateEnv(s_id, service_path).get_env()

    def _put_attrs(self, client_func, s_id, content, comment):
        content.pop('change_info')
        content['snapshot_id'] = content.pop('_id')
        content['comment'] = comment
        return client_func(s_id, content)


class RuntimeAttrsCtrl(AttrsCtrl):
    aspect_name = 'runtime_attrs'
    template_name = 'template_runtime_attrs.yml'
    rendered_name = 'runtime_attrs.yml'
    raw_content_name = 'raw_runtime_attrs.json'

    def get_attrs(self, s_id):
        return self.client.get_runtime_attrs(s_id)

    def put_attrs(self, s_id, content, comment):
        client_func = self.client.put_runtime_attrs
        return self._put_attrs(client_func, s_id, content, comment)

    def convert_to_yml(self, service_path, content):
        content['content']['resources'] = ResourceManager.convert_and_save_resources(
            service_path,
            content['content']['resources']
        )
        return content

    def convert_to_json(self, service_path, content):
        content['content']['resources'] = ResourceManager.retrieve_and_convert_resources(
            content['content']['resources']
        )
        return content

    def get_template_env(self, s_id, service_path):
        return template_env.RuntimeAttrsEnv(s_id, service_path).get_env()


class InfoAttrsCtrl(AttrsCtrl):
    aspect_name = 'info_attrs'
    template_name = 'template_info_attrs.yml'
    rendered_name = 'info_attrs.yml'
    raw_content_name = 'raw_info_attrs.json'

    def get_attrs(self, s_id):
        return self.client.get_info_attrs(s_id)

    def put_attrs(self, s_id, content, comment):
        client_func = self.client.put_info_attrs
        return self._put_attrs(client_func, s_id, content, comment)

    def convert_to_json(self, service_path, content):
        return content

    def convert_to_yml(self, service_path, content):
        return content


class AuthAttrsCtrl(AttrsCtrl):
    aspect_name = 'auth_attrs'
    template_name = 'template_auth_attrs.yml'
    rendered_name = 'auth_attrs.yml'
    raw_content_name = 'raw_auth_attrs.json'

    def get_attrs(self, s_id):
        return self.client.get_auth_attrs(s_id)

    def put_attrs(self, s_id, content, comment):
        client_func = self.client.put_auth_attrs
        return self._put_attrs(client_func, s_id, content, comment)

    def convert_to_json(self, service_path, content):
        return content

    def convert_to_yml(self, service_path, content):
        return content
