from __future__ import unicode_literals

import json
import os

import yaml
from sepelib.util.fs import remove_ignore

from service_repo_client.lib.errors import InvalidContentError, VersionConflictError


class AspectOperator(object):
    def __init__(self, service_id, aspect_ctrl, dir_path):
        """
        :type service_id: unicode
        :type aspect_ctrl: service_repo_client.aspect_ctrl.IAspectCtrl
        :type dir_path: unicode
        """
        self.service_id = service_id
        self.aspect_ctrl = aspect_ctrl
        self.dir_path = dir_path
        self.service_path = os.path.join(self.dir_path, self.service_id)

    def path_to_file(self, filename):
        return os.path.join(self.service_path, filename)

    def checkout(self, save=True):
        """
        :type save: bool
        :rtype: dict
        """
        content = self.aspect_ctrl.get_attrs(self.service_id)
        if content and save:
            self.save_content(content)
        return content

    def save_content(self, content):
        self._save_to_json(self.service_id, content)
        content = self.aspect_ctrl.convert_to_yml(self.service_path, content)
        self._save_to_yml(self.path_to_file(self.aspect_ctrl.rendered_name), content)
        self._save_to_yml(self.path_to_file(self.aspect_ctrl.template_name), content)

    def _save_to_json(self, s_id, data):
        raw_content_name = self.aspect_ctrl.raw_content_name
        with open(self.path_to_file(raw_content_name), 'w') as fd:
            json.dump(data, fd, indent=4, sort_keys=True)

    def _save_to_yml(self, filename, data):
        with open(filename, 'w') as fd:
            yaml.safe_dump(data=data, stream=fd, default_flow_style=False, indent=4)

    def checkout_force(self):
        content = self.aspect_ctrl.get_attrs(self.service_id)
        current_version = self._get_current_version()
        if self.aspect_ctrl.get_version(content) != current_version:
            print 'Updating {}...'.format(self.aspect_ctrl.aspect_name)
            if content:
                self.save_content(content)
            else:
                self._remove_all()

    def _get_current_version(self):
        content = self.get_json_content()
        return self.aspect_ctrl.get_version(content) if content else None

    def get_json_content(self):
        raw_content_name = self.path_to_file(self.aspect_ctrl.raw_content_name)
        if not os.path.exists(raw_content_name):
            return None
        try:
            with open(raw_content_name) as fd:
                content = json.load(fd)
        except ValueError as err:
            raise InvalidContentError('Could not parse {}:\n{}'.format(raw_content_name, err.message))
        return content

    def _remove_all(self):
        remove_ignore(self.path_to_file(self.aspect_ctrl.template_name))
        remove_ignore(self.path_to_file(self.aspect_ctrl.rendered_name))
        remove_ignore(self.path_to_file(self.aspect_ctrl.raw_content_name))

    def check_version(self):
        content = self.aspect_ctrl.get_attrs(self.service_id)
        version = self.aspect_ctrl.get_version(content)
        current_version = self._get_current_version()
        if version != current_version:
            raise VersionConflictError('Newer version of {} found.'.format(self.aspect_ctrl.aspect_name))

    def build(self, context):
        """
        :type context: dict
        """
        template = self.path_to_file(self.aspect_ctrl.template_name)
        if not os.path.exists(template):
            return
        rendered = self._render_service_from_template(self.aspect_ctrl.template_name, context)
        try:
            content = yaml.load(rendered)
        except yaml.YAMLError as err:
            raise InvalidContentError('Could not parse {}:\n{}'.format(template, err))
        self._save_to_yml(self.path_to_file(self.aspect_ctrl.rendered_name), content)
        content = self.aspect_ctrl.convert_to_json(self.service_path, content)
        self._save_to_json(self.service_id, content)

    def _render_service_from_template(self, filename, context):
        env = self.aspect_ctrl.get_template_env(self.service_id, self.service_path)
        template = env.get_template(filename)
        return template.render(context)

    def upload(self, comment):
        """
        :type comment: unicode
        """
        content = self.get_json_content()
        if not content:
            return
        uploaded_content = self.aspect_ctrl.put_attrs(self.service_id, content=content, comment=comment)
        # save content with new id/version
        self.save_content(uploaded_content)
