import copy
import os
import shutil

from sepelib.util.fs import makedirs_ignore

from service_repo_client.lib.io_helper import FileOperator, JsonFileOperator


class FilesManager(object):
    file_operator = NotImplemented

    @classmethod
    def get(cls, section_path, resources):
        """
        :type section_path: str | unicode
        :type resources: list
        :rtype: dict
        """
        raise NotImplementedError

    @classmethod
    def put(cls, section, resources):
        """
        :type section: str | unicode
        :type resources: dict
        :rtype: list
        """
        raise NotImplementedError


class NonStoredFiles(FilesManager):
    @classmethod
    def get(cls, section_path, resources):
        result = {}
        for resource in resources:
            local_path = resource.pop('local_path')
            result[local_path] = resource
        return result

    @classmethod
    def put(cls, section, resources):
        result = []
        for local_path, resource in resources.iteritems():
            resource['local_path'] = local_path
            result.append(resource)
        return result


class StoredFiles(FilesManager):
    file_operator = FileOperator

    @classmethod
    def get(cls, section_path, resources):
        result = {}
        if os.path.exists(section_path):
            shutil.rmtree(section_path)
        for resource in resources:
            local_path = resource.pop('local_path')
            result[local_path] = resource
            file_path = os.path.join(section_path, local_path)
            makedirs_ignore(os.path.dirname(file_path))
            cls.file_operator.write(file_path, resource['content'])
            resource['content'] = file_path
        return result

    @classmethod
    def put(cls, section, resources):
        result = []
        for local_path, resource in resources.iteritems():
            resource['local_path'] = local_path
            resource['content'] = cls.file_operator.read(resource['content'])
            result.append(resource)
        return result


class StoredJsonFiles(StoredFiles):
    file_operator = JsonFileOperator


class TemplateSetFiles(FilesManager):
    file_operator = FileOperator
    ROOT_TEMPLATE_NAME = 'root_template'
    TEMPLATE_FOLDER_NAME = 'files'

    @classmethod
    def get(cls, section_path, resources):
        result = {}
        if os.path.exists(section_path):
            shutil.rmtree(section_path)
        for resource in resources:
            local_path = resource.pop('local_path')
            result[local_path] = resource
            makedirs_ignore(os.path.join(section_path, local_path, cls.TEMPLATE_FOLDER_NAME))
            file_path = os.path.join(section_path, local_path, cls.ROOT_TEMPLATE_NAME)
            cls.file_operator.write(file_path, resource['layout'])
            resource['layout'] = file_path
            for template_file in resource['templates']:
                template_name = template_file['name']
                file_path = os.path.join(section_path, local_path, 'files', template_name)
                cls.file_operator.write(file_path, template_file['content'])
                template_file['content'] = file_path
        return result

    @classmethod
    def put(cls, section, resources):
        result = []
        for local_path, resource in resources.iteritems():
            resource['local_path'] = local_path
            resource['layout'] = cls.file_operator.read(resource['layout'])
            templates = []
            for template_file in resource['templates']:
                templates.append({
                    'name': template_file['name'],
                    'content': cls.file_operator.read(template_file['content'])
                })
            resource['templates'] = templates
            result.append(resource)
        return result


class ResourceManager(object):
    files_managers = {
        'sandbox_files': NonStoredFiles,
        'url_files': NonStoredFiles,
        'static_files': StoredFiles,
        'template_set_files': TemplateSetFiles,
        'l7_fast_balancer_config_files': StoredJsonFiles,
        'services_balancer_config_files': StoredJsonFiles
    }

    @classmethod
    def convert_and_save_resources(cls, service_path, resources):
        """
        :type service_path: str | unicode
        :type resources: dict
        :rtype: dict
        """
        sections = copy.deepcopy(resources)
        for section, manager_cls in cls.files_managers.iteritems():
            section_resources = sections.get(section)
            if section_resources is not None:
                section_path = os.path.join(service_path, section)
                sections[section] = manager_cls.get(section_path, section_resources)
        return sections

    @classmethod
    def retrieve_and_convert_resources(cls, resources):
        """
        :type resources: dict
        :rtype: dict
        """
        sections = copy.deepcopy(resources)
        for section, manager_cls in cls.files_managers.iteritems():
            section_resources = sections.get(section)
            if section_resources is not None:
                sections[section] = manager_cls.put(section, section_resources)
        return sections
