from __future__ import print_function
from __future__ import unicode_literals

import argparse
import logging
import errno
import os
import sys
import traceback
from urlparse import urljoin

from nanny_repo import repo_api_stub

import nanny_rpc_client

from infra.nanny.nanny_services_rest.nanny_services_rest import client as nanny_client
from infra.nanny.nanny_services_rest.nanny_services_rest import errors as client_errors

import service_repo_client.actions.build
import service_repo_client.actions.dump
import service_repo_client.actions.get
import service_repo_client.actions.list
import service_repo_client.actions.restore
import service_repo_client.actions.update
import service_repo_client.actions.upload
import service_repo_client.actions.checkout_v2
import service_repo_client.actions.upload_v2
from sepelib.core import config
from service_repo_client.cfg import get_default_config
from service_repo_client.lib.errors import ServiceRepoClientError, VersionConflictError, AuthError
from service_repo_client.manager import ServiceRepoManager
from service_repo_client.version import VERSION


NANNY_URL = 'https://nanny.yandex-team.ru'
DEV_NANNY_URL = 'https://dev-nanny.yandex-team.ru'
TOKEN_PATH = '~/.scli/token'
DEFAULT_CONFIG_PATH = '~/.scli/config.yml'
NANNY_TOKEN_ENV_VAR = 'NANNY_OAUTH_TOKEN'


def parse_args():
    parser = argparse.ArgumentParser('Service repo client')
    parser.add_argument('-u', '--url', help='Use specified Nanny URL')
    parser.add_argument('-T', '--testing', action='store_true', help='Use testing Nanny installation')
    parser.add_argument('-t', '--token', metavar='OAUTH_TOKEN',
                        help='Nanny OAuth token (default: get from {})'.format(TOKEN_PATH))
    parser.add_argument('-v', '--version', action='version', version=VERSION)
    parser.add_argument('-c', '--cfg',
                        default=None,
                        action='store',
                        help='path to service cfg file (default: get from {})'.format(DEFAULT_CONFIG_PATH))
    parser.add_argument('-d', '--dir',
                        default='.',
                        action='store',
                        help='path to working dir')

    subparsers = parser.add_subparsers(title='Possible actions')

    service_repo_client.actions.list.add_parsers(subparsers)
    service_repo_client.actions.get.add_parsers(subparsers)
    service_repo_client.actions.checkout_v2.add_parsers(subparsers)
    service_repo_client.actions.build.add_parsers(subparsers)
    service_repo_client.actions.update.add_parsers(subparsers)
    service_repo_client.actions.upload.add_parsers(subparsers)
    service_repo_client.actions.upload_v2.add_parsers(subparsers)
    service_repo_client.actions.dump.add_parsers(subparsers)
    service_repo_client.actions.restore.add_parsers(subparsers)

    return parser.parse_args()


def _get_token():
    config_path = os.path.expanduser(TOKEN_PATH)
    try:
        with open(config_path) as fd:
            return fd.read().strip()
    except EnvironmentError as e:
        if e.errno != errno.ENOENT:
            raise


def _load_config(args):
    if args.cfg:
        config_path = args.cfg
    else:
        config_path = os.path.expanduser(DEFAULT_CONFIG_PATH)
        if not os.path.exists(config_path):
            config_path = None
    config.load(config_path)
    config.merge(get_default_config())


def main():
    logging.basicConfig(
        stream=sys.stderr,
        level=logging.INFO,
        format='%(asctime)s\t%(levelname)s\t%(module)s\t%(funcName)s\t%(lineno)d\t%(message)s',
    )

    try:
        # process command line arguments
        args = parse_args()

        token = args.token or os.environ.data.get(NANNY_TOKEN_ENV_VAR) or _get_token()
        if token is None:
            msg = 'OAuth token is not given in -t/--token argument and is not present in {}'.format(TOKEN_PATH)
            raise AuthError(msg)
        if args.url is not None:
            url = args.url
        elif args.testing:
            url = DEV_NANNY_URL
        else:
            url = NANNY_URL

        _load_config(args)
        client = nanny_client.ServiceRepoClient(url, token, **config.get_value('service_repo_client'))
        rpc_client = nanny_rpc_client.RetryingRpcClient(urljoin(url, '/api/repo'), token)
        repo_stub = repo_api_stub.RepoServiceStub(rpc_client)
        repo_manager = ServiceRepoManager(client, repo_stub, args.dir)
        args.handle(args, repo_manager)
    # some handy exception handling
    # to show nicely formatted and detailed enough
    # error messages to user
    except SystemExit:
        raise
    except KeyboardInterrupt:
        print("exiting on user request", file=sys.stderr)
        exit_code = 0
    except AuthError as e:
        print(unicode(e), file=sys.stderr)
        exit_code = 3
    except (VersionConflictError, client_errors.ModificationConflictError) as e:
        print(unicode(e), file=sys.stderr)
        exit_code = 4
    except client_errors.ServiceRepoRequestError as e:
        print(unicode(e), file=sys.stderr)
        exit_code = 5
    except ServiceRepoClientError as e:
        print(unicode(e), file=sys.stderr)
        exit_code = 6
    except client_errors.NotFoundError as e:
        print(unicode(e), file=sys.stderr)
        exit_code = 7
    except Exception as e:
        print(e, file=sys.stderr)
        traceback.print_exc()
        exit_code = 1
    else:
        exit_code = 0

    raise SystemExit(exit_code)
