from __future__ import unicode_literals

import errno
import json
import os

from nanny_repo import repo_api_pb2

from service_repo_client import aspect_ctrl
from lib.errors import ServiceCheckoutError, ServiceDirNotFoundError, ServiceExistsError
from service_repo_client.aspect_operator import AspectOperator
import service_repo_client.upload_v2_controller.controller as upload_v2_controller


class IServiceRepoManager(object):

    def list(self, login=None):
        """
        :type login: str | unicode | None
        :rtype list[str]
        """
        raise NotImplementedError

    def checkout(self, s_id, force, stdout, pretty):
        """
        :type s_id: unicode
        :type force: bool
        :type stdout: bool
        :type pretty: bool
        """
        raise NotImplementedError

    def build(self, s_id, context):
        """
        :type s_id: unicode
        :type context: dict
        """
        raise NotImplementedError

    def upload(self, s_id, comment):
        """
        :type s_id: unicode
        :type comment: unciode
        """
        raise NotImplementedError

    def update(self, s_id):
        """
        :type s_id: unicode
        """
        raise NotImplementedError

    def create(self, s_id, comment):
        """
        :type s_id: unicode
        :type comment: unicode
        """
        raise NotImplementedError


class ServiceRepoManager(IServiceRepoManager):
    DEFAULT_JSON_FORMAT = {
        'indent': 4,
        'separators': (',', ': ')
    }

    def __init__(self, client, repo_stub, dir_path):
        """
        :type client: nanny_services_rest.client.ServiceRepoClient
        :type repo_stub: nanny_repo.repo_api_stub.RepoServiceStub
        :type dir_path: unicode
        """
        self.client = client
        self.json_format = self.DEFAULT_JSON_FORMAT
        self._aspect_controllers = None
        self.repo_stub = repo_stub
        self.dir_path = dir_path

    def _init_ascpect_controllers(self):
        self._aspect_controllers = {
            'cleanup_policy': aspect_ctrl.CleanupPolicyCtrl(self.repo_stub),
            'auth_attrs': aspect_ctrl.AuthAttrsCtrl(self.client),
            'runtime_attrs': aspect_ctrl.RuntimeAttrsCtrl(self.client),
            'info_attrs': aspect_ctrl.InfoAttrsCtrl(self.client),
        }

    @property
    def aspect_controllers(self):
        if not self._aspect_controllers:
            self._init_ascpect_controllers()
        return self._aspect_controllers.values()

    @property
    def aspect_controllers_dict(self):
        if not self._aspect_controllers:
            self._init_ascpect_controllers()
        return self._aspect_controllers

    def _must_create_service_dir(self, s_id):
        """
        :type s_id: unicode
        """
        try:
            os.mkdir(os.path.join(self.dir_path, s_id))
        except OSError as e:
            if e.errno == errno.EEXIST:
                raise ServiceExistsError('Service dir "{0}" exists already'.format(s_id))
            raise ServiceCheckoutError('Cannot checkout "{0}": {1}'.format(s_id, e))
        except Exception as e:
            raise ServiceCheckoutError('Cannot checkout "{0}": {1}'.format(s_id, e))

    def _service_must_exist(self, s_id):
        if not os.path.isdir(os.path.join(self.dir_path, s_id)):
            raise ServiceDirNotFoundError('Service "{0}" has not been checked out'.format(s_id))

    def _check_version(self, s_id):
        for ctrl in self.aspect_controllers:
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            operator.check_version()

    def list(self, login=None):
        req = repo_api_pb2.ListSummariesRequest()
        if login:
            req.for_login = login
        services = self.repo_stub.list_summaries(req).value
        return [s.service_id for s in services]

    def checkout(self, s_id, force, stdout, pretty):
        save_to_file = not stdout and not pretty
        if save_to_file:
            try:
                self._must_create_service_dir(s_id)
            except ServiceExistsError:
                if not force:
                    raise
        result = {}
        for ctrl in self.aspect_controllers:
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            result[ctrl.aspect_name] = operator.checkout(save=save_to_file)
        if pretty:
            print(json.dumps(result, **self.json_format))
        elif stdout:
            print(json.dumps(result))

    def build(self, s_id, context):
        self._service_must_exist(s_id)
        for ctrl in self.aspect_controllers:
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            operator.build(context)

    def upload(self, s_id, comment):
        self._service_must_exist(s_id)
        self._check_version(s_id)
        for ctrl in self.aspect_controllers:
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            operator.upload(comment)

    def update(self, s_id):
        self._service_must_exist(s_id)
        for ctrl in self.aspect_controllers:
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            operator.checkout_force()

    def create(self, s_id, comment):
        self._service_must_exist(s_id)
        new_service_aspects = ['info_attrs', 'auth_attrs', 'runtime_attrs']
        content = {
            'id': s_id,
            'comment': comment
        }
        for aspect_name in new_service_aspects:
            ctrl = self.aspect_controllers_dict[aspect_name]
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            content[aspect_name] = operator.get_json_content()['content']
        self.client.create_service(content)

        # Update attrs versions
        for aspect_name in new_service_aspects:
            ctrl = self.aspect_controllers_dict[aspect_name]
            operator = AspectOperator(s_id, ctrl, self.dir_path)
            operator.checkout_force()

        # Modify newly created cleanup policy
        ctrl = self.aspect_controllers_dict['cleanup_policy']
        operator = AspectOperator(s_id, ctrl, self.dir_path)
        new_content = operator.checkout(save=False)
        local_content = operator.get_json_content()
        if local_content:
            new_content['policy']['spec'] = local_content['policy']['spec']
        operator.save_content(new_content)
        operator.upload(comment)

    def checkout_v2(self, s_id):
        self.checkout(s_id, force=True, stdout=False, pretty=False)
        controller = upload_v2_controller.MainController(s_id, self.dir_path)
        controller.save_nanny_state()

    def upload_v2(self, s_id, comment, scheduling_priority, colored_diff, auto_yes):
        self.checkout(s_id, force=True, stdout=False, pretty=False)
        controller = upload_v2_controller.MainController(s_id, self.dir_path)
        if controller.process(colored_diff, auto_yes):
            controller.update_meta_info(comment, scheduling_priority)
            self.build(s_id, {})
            self.upload(s_id, comment)
