import click
import difflib
import os
import cStringIO
import yaml
import logging

INFO_ATTRS = 'info_attrs'
RUNTIME_ATTRS = 'runtime_attrs'
AUTH_ATTRS = 'auth_attrs'

FILES_CACHE = {}


def _dump_to_str(data):
    fd = cStringIO.StringIO()
    yaml.safe_dump(data=data, stream=fd, default_flow_style=False, indent=2, sort_keys=True)
    result = fd.getvalue()
    fd.close()
    return result


def _get_yaml_diff(old, new):
    left_lines = _dump_to_str(old).splitlines()
    right_lines = _dump_to_str(new).splitlines()

    return list(difflib.unified_diff(left_lines, right_lines, fromfile='before', tofile='after', lineterm=''))


def _color_line(line, color, colored):
    if not colored:
        return line
    else:
        return click.style(line, fg=color)


def print_diff(colored_diff, service_id, old, new):
    diff_lines = _get_yaml_diff(old, new)
    if not diff_lines:
        logging.info(_color_line('No diff for {}'.format(service_id), color='yellow', colored=colored_diff))
        return False

    logging.info(_color_line('Diff for {}'.format(service_id), color='yellow', colored=colored_diff))
    for line in diff_lines:
        line = line.rstrip()
        if line.startswith('-'):
            logging.info(_color_line(line, color='red', colored=colored_diff))
        elif line.startswith('+'):
            logging.info(_color_line(line, color='green', colored=colored_diff))
        else:
            logging.info(line)
    return True


def load_yaml(path):
    with open(path) as fd:
        return yaml.safe_load(fd)


def load_template(d, t):
    path = os.path.join(d, 'template_{}.yml'.format(t))
    if path not in FILES_CACHE:
        FILES_CACHE[path] = load_yaml(os.path.join(d, 'template_{}.yml'.format(t)))
    return FILES_CACHE[path]


def dump_yaml(path, data):
    with open(path, 'w') as fd:
        yaml.safe_dump(data=data, stream=fd, default_flow_style=False, indent=2)


def dump_template(d, t, data):
    path = os.path.join(d, 'template_{}.yml'.format(t))
    FILES_CACHE[path] = data


def dump_cache():
    global FILES_CACHE
    for k, v in FILES_CACHE.items():
        dump_yaml(k, v)
    FILES_CACHE = {}


class BaseController(object):
    def load_state(self, state, working_directory):
        raise NotImplementedError('NotImplemented')

    def update_state_before_diff(self, old_state, new_state):
        return

    def apply_state(self, state, working_directory):
        raise NotImplementedError('NotImplemented')


class FieldMapping(object):
    NOTHING = '__nothing'

    def __init__(self, state_key, template_key, default=NOTHING):
        self.state_key = state_key
        self.template_key = template_key
        self.default = default


def _iter_over_mapping(mapping):
    for m in mapping:
        if isinstance(m, FieldMapping):
            state_key = m.state_key
            template_key = m.template_key
            default = mapping.default
        else:
            state_key = m[0]
            template_key = m[1]
            if len(m) > 2:
                default = m[2]
            else:
                default = FieldMapping.NOTHING

        yield (state_key, template_key, default)


def from_template_to_state_by_mapping(template, mapping):
    result = {}
    for state_key, template_key, default in _iter_over_mapping(mapping):
        if default == FieldMapping.NOTHING:
            result[state_key] = template[template_key]
        else:
            if template[template_key] != default:
                result[state_key] = template[template_key]

    return result


def from_state_to_template_by_mapping(state, mapping):
    result = {}
    for state_key, template_key, default in _iter_over_mapping(mapping):
        if default != FieldMapping.NOTHING:
            result[template_key] = state.get(state_key, default)
        else:
            result[template_key] = state[state_key]

    return result
