from __future__ import unicode_literals

import logging
import os
import re
import functools

import service_repo_client.upload_v2_controller.common as lib_common
import service_repo_client.upload_v2_controller.auth_attrs_controllers as auth_attrs_controllers
import service_repo_client.upload_v2_controller.info_attrs_controllers as info_attrs_controllers
import service_repo_client.upload_v2_controller.runtime_attrs_controllers as runtime_attrs_controllers

CATEGORIES_PREFIX_WHITE_LIST = (
    '/ydo/',
    '/ydo_test/',
)


def dump_cache_wrapper(f):
    @functools.wraps(f)
    def wrapped(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        finally:
            lib_common.dump_cache()

    return wrapped


class MainController(object):
    controllers = [
        info_attrs_controllers.CategoryController,
        info_attrs_controllers.AbcGroupController,
        info_attrs_controllers.DescriptionController,
        info_attrs_controllers.QueueController,
        info_attrs_controllers.PrepareRecipesController,
        info_attrs_controllers.ActivateRecipesController,
        info_attrs_controllers.SchedulingPolicyController,
        info_attrs_controllers.SandboxResourcesReleasesController,
        info_attrs_controllers.DeployMonitoringController,
        info_attrs_controllers.InfraController,

        runtime_attrs_controllers.StaticFilesController,
        runtime_attrs_controllers.SandboxResourcesController,
        runtime_attrs_controllers.InitContainersController,

        runtime_attrs_controllers.TemplateVolumesController,
        runtime_attrs_controllers.VaultSecretVolumesController,
        runtime_attrs_controllers.ItsVolumesController,
        runtime_attrs_controllers.NannySecretVolumesController,

        runtime_attrs_controllers.RuntimeContainersController,

        auth_attrs_controllers.OwnersController,
        auth_attrs_controllers.ConfigurationManagersController,
        auth_attrs_controllers.OperationManagersController,
        auth_attrs_controllers.ObserversController,
    ]

    def __init__(self, s_id, working_dir):
        self.service_id = s_id
        self.working_dir = working_dir
        self.service_dir = os.path.join(working_dir, s_id)

    def load_current_state(self):
        state = {'id': self.service_id}
        for controller in self.controllers:
            controller_instance = controller()
            controller_instance.load_state(state, self.service_dir)
        category = state['category']
        allowed = False
        for category_prefix in CATEGORIES_PREFIX_WHITE_LIST:
            if category.startswith(category_prefix):
                allowed = True
                break
        if not allowed:
            raise RuntimeWarning('Sorry, you can\'t use this mode. '
                                 'Please connect with zhshishkin@ if you need it. '
                                 'https://st.yandex-team.ru/YDO-15075')
        return state

    def save_nanny_state(self):
        current_nanny_state = self.load_current_state()
        lib_common.dump_yaml('{}.yaml'.format(self.service_id), current_nanny_state)

    def apply_state(self, state):
        for controller in self.controllers:
            controller_instance = controller()
            controller_instance.apply_state(state, self.service_dir)

    def update_state_before_diff(self, nanny_state, local_state):
        for controller in self.controllers:
            controller_instance = controller()
            controller_instance.update_state_before_diff(nanny_state, local_state)

    @dump_cache_wrapper
    def process(self, colored_diff, auto_yes):
        current_nanny_state = self.load_current_state()
        local_state = lib_common.load_yaml('{}.yaml'.format(self.service_id))
        self.update_state_before_diff(current_nanny_state, local_state)
        if lib_common.print_diff(colored_diff, self.service_id, current_nanny_state, local_state):
            if not auto_yes:
                var = raw_input('Please enter yes to apply\n')
                if var[0].lower() != 'y':
                    logging.info('Skip applying')
                    return

            logging.info('Apply state')
            self.apply_state(local_state)
            return True
        else:
            return False

    STARTREK_TICKETS_RE = re.compile('[A-Z]+-[1-9][0-9]*')

    @dump_cache_wrapper
    def update_meta_info(self, comment, scheduling_priority):
        template = lib_common.load_template(self.service_dir, lib_common.RUNTIME_ATTRS)
        template['meta_info'] = {
            'scheduling_config': {'scheduling_priority': scheduling_priority},
            'startrek_tickets': list(set(self.STARTREK_TICKETS_RE.findall(comment))),
            'is_disposable': False,
        }
        lib_common.dump_template(self.service_dir, lib_common.RUNTIME_ATTRS, template)
