import service_repo_client.upload_v2_controller.common as lib_common
import service_repo_client.upload_v2_controller.storages as lib_storages


def _get(d, p):
    for i in p[:-1]:
        d = d.get(i, {})
    return d.get(p[-1], None)


def _set(d, p, v):
    for i in p[:-1]:
        d.setdefault(i, dict())
        d = d[i]
    d[p[-1]] = v


class SimpleController(lib_common.BaseController):
    state_path = None
    template_path = None
    template_type = None
    optional = False

    def load_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, self.template_type)
        value = _get(template, self.template_path)
        if self.optional and not value:
            return
        _set(state, self.state_path, value)

    def apply_state(self, state, working_dir):
        value = _get(state, self.state_path)
        if self.optional and not value:
            return
        template = lib_common.load_template(working_dir, self.template_type)
        _set(template, self.template_path, value)
        lib_common.dump_template(working_dir, self.template_type, template)


class CategoryController(SimpleController):
    state_path = ['category']
    template_path = ['content', 'category']
    template_type = lib_common.INFO_ATTRS


class AbcGroupController(SimpleController):
    state_path = ['abc']
    template_path = ['content', 'abc_group']
    template_type = lib_common.INFO_ATTRS


class DescriptionController(SimpleController):
    state_path = ['description']
    template_path = ['content', 'desc']
    template_type = lib_common.INFO_ATTRS


class QueueController(SimpleController):
    state_path = ['queue']
    template_path = ['content', 'queue_id']
    template_type = lib_common.INFO_ATTRS
    optional = True


class RecipesController(lib_common.BaseController):
    template_recipes_section = None
    state_recipes_section = None

    def load_state(self, state, working_dir):
        result = dict()
        recipes = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)['content']['recipes'][self.template_recipes_section]
        if recipes:
            for recipe in recipes:
                result[recipe['id']] = {
                    'name': recipe['name'],
                    'description': recipe['desc'],
                    'context': {value['key']: value['value'] for value in recipe['context']}
                }
                if recipe.get('labels'):
                    result['labels'] = recipe['labels']

            state.setdefault('recipes', {})[self.state_recipes_section] = result

    def apply_state(self, state, working_dir):
        recipes = state.get('recipes', {}).get(self.state_recipes_section, {})
        template_result = []
        for recipe_id in sorted(recipes.keys()):
            recipe = recipes[recipe_id]
            template_result.append(
                {
                    'id': recipe_id,
                    'name': recipe['name'],
                    'desc': recipe['description'],
                    'context': [{'key': k, 'value': recipe['context'][k]} for k in sorted(recipe['context'].keys())],
                    'labels': recipe.get('labels', [])
                }
            )

        template = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)
        template['content']['recipes'][self.template_recipes_section] = template_result
        lib_common.dump_template(working_dir, lib_common.INFO_ATTRS, template)


class PrepareRecipesController(RecipesController):
    template_recipes_section = 'prepare_recipes'
    state_recipes_section = 'prepare'


class ActivateRecipesController(RecipesController):
    template_recipes_section = 'content'
    state_recipes_section = 'activate'


class SchedulingPolicyController(lib_common.BaseController):
    def load_state(self, state, working_dir):
        scheduling_policy = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)['content']['scheduling_policy']

        t = scheduling_policy.get('type', 'NONE')
        if t != 'NONE':
            t_lower = t.lower()
            state['scheduling_policy'] = {
                'type': t,
                'activate_recipe': scheduling_policy[t_lower]['activate_recipe'],
                'prepare_recipe': scheduling_policy[t_lower]['prepare_recipe'],
            }

    def apply_state(self, state, working_dir):
        if 'scheduling_policy' not in state or state['scheduling_policy']['type'] == 'NONE':
            scheduling_policy = {'type': 'NONE'}
        else:
            scheduling_policy = {
                'type': state['scheduling_policy']['type'],
                state['scheduling_policy']['type'].lower(): {
                    'activate_recipe': state['scheduling_policy']['activate_recipe'],
                    'prepare_recipe': state['scheduling_policy']['prepare_recipe'],

                }
            }

        template = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)
        template['content']['scheduling_policy'] = scheduling_policy
        lib_common.dump_template(working_dir, lib_common.INFO_ATTRS, template)


class SandboxResourcesReleasesController(lib_common.BaseController):
    optional_fields = [
        ('queue_id', 'queue'),
        ('responsibles', 'responsible_persons'),
        ('sandbox_resource_type', 'resource_type'),
        ('sandbox_task_type', 'task_type'),
    ]
    not_none_scheduling_optional_fields = [
        ('sched_activate_recipe', 'activate_recipe'),
        ('sched_prepare_recipe', 'prepare_recipe'),
    ]

    def load_one_release_rule(self, release_rule):
        state = {
            'description': release_rule['desc'],
            'expression': release_rule['filter_params']['expression'],
            'ticket_priority': release_rule['ticket_priority'],
        }
        for template_field, state_field in self.optional_fields:
            if release_rule.get(template_field):
                state[state_field] = release_rule[template_field]
        if release_rule.get('auto_commit_settings', {}).get('enabled'):
            scheduling = {'priority': release_rule['auto_commit_settings'].get('scheduling_priority', 'NONE')}
            if scheduling['priority'] != 'NONE':
                for template_field, state_field in self.not_none_scheduling_optional_fields:
                    if release_rule.get(template_field):
                        scheduling[state_field] = release_rule[template_field]
            state['scheduling'] = scheduling
        if release_rule.get('approve_policy', {}).get('type', 'NONE') != 'NONE':
            approve_policy = {'type': release_rule['approve_policy']['type']}
            assert approve_policy['type'] == 'MULTIPLE_APPROVE'
            multiple_approve = release_rule['approve_policy']['multiple_approve']
            approve_policy['approves_count'] = multiple_approve['approves_count']
            approve_policy['approvers'] = {}

            if multiple_approve['approvers'].get('logins'):
                approve_policy['approvers']['logins'] = multiple_approve['approvers']['logins']
            if multiple_approve['approvers'].get('groups'):
                groups_storage = lib_storages.GroupsStorage()
                approve_policy['approvers']['groups'] = [groups_storage.get_url_by_id(group_id) for group_id in multiple_approve['approvers']['groups']]
            if multiple_approve['mandatory_approvers'].get('logins'):
                approve_policy['approvers']['mandatory_approvers'] = multiple_approve['mandatory_approvers']['logins']

            state['approve_policy'] = approve_policy

        return state

    def load_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)['content']['tickets_integration']
        if not template.get('service_release_tickets_enabled'):
            return
        release_rules = template['service_release_rules']
        state_rules = [self.load_one_release_rule(release_rule) for release_rule in release_rules]
        if state_rules:
            state['sandbox_release_rules'] = state_rules

    def apply_one_release_rule(self, state):
        release_rule = {
            'desc': state['description'],
            'filter_params': {'expression': state['expression']},
            'ticket_priority': state['ticket_priority'],
        }
        for template_field, state_field in self.optional_fields:
            if state.get(state_field):
                release_rule[template_field] = state[state_field]
        if state.get('scheduling'):
            scheduling = state['scheduling']
            if scheduling['priority'] != 'NONE':
                for template_field, state_field in self.not_none_scheduling_optional_fields:
                    if scheduling.get(state_field):
                        release_rule[template_field] = scheduling[state_field]
            release_rule['auto_commit_settings'] = {'enabled': True, 'scheduling_priority': scheduling['priority']}
        else:
            release_rule['auto_commit_settings'] = {'enabled': False}

        if state.get('approve_policy'):
            approve_policy = state['approve_policy']
            groups_storage = lib_storages.GroupsStorage()
            release_rule['approve_policy'] = {
                'type': approve_policy['type'],
                'multiple_approve': {
                    'approves_count': approve_policy['approves_count'],
                    'approvers': {
                        'logins': approve_policy['approvers'].get('logins', []),
                        'groups': [groups_storage.get_id_by_url(url) for url in approve_policy['approvers'].get('groups', [])],
                    },
                    'mandatory_approvers': {
                        'logins': approve_policy['approvers'].get('mandatory_approvers', []),
                    }
                }
            }
        else:
            release_rule.pop('approve_policy', None)

        return release_rule

    def apply_state(self, state, working_dir):
        if 'sandbox_release_rules' not in state:
            return

        template = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)
        template['content']['tickets_integration']['service_release_tickets_enabled'] = True
        template['content']['tickets_integration']['service_release_rules'] = [self.apply_one_release_rule(rule_state) for rule_state in state['sandbox_release_rules']]
        lib_common.dump_template(working_dir, lib_common.INFO_ATTRS, template)


class DeployMonitoringController(lib_common.BaseController):
    def load_state(self, state, working_dir):
        deploy_monitoring = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)['content']['monitoring_settings'].get('deploy_monitoring', {})

        if deploy_monitoring.get('is_enabled', False):
            result = {
                'timeout': deploy_monitoring['content']['deploy_timeout'],
                'methods': deploy_monitoring['content']['alert_methods'],
                'logins': deploy_monitoring['content']['responsible']['logins']
            }
            state.setdefault('monitoring', dict())['deploy'] = result

    def apply_state(self, state, working_dir):
        deploy_monitoring_state = state.get('monitoring', dict()).get('deploy')
        if deploy_monitoring_state:
            deploy_monitoring = {
                'is_enabled': True,
                'content': {
                    'alert_methods': deploy_monitoring_state['methods'],
                    'responsible': {
                        'logins': deploy_monitoring_state['logins'],
                        'groups': []
                    },
                    'deploy_timeout': deploy_monitoring_state['timeout'],
                }
            }
        else:
            deploy_monitoring = {'is_enabled': False}

        template = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)
        template['content'].setdefault('monitoring_settings', {})['deploy_monitoring'] = deploy_monitoring
        lib_common.dump_template(working_dir, lib_common.INFO_ATTRS, template)


class InfraController(lib_common.BaseController):
    def load_state(self, state, working_dir):
        infra_notifications = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)['content'].get('infra_notifications', {})

        if infra_notifications:
            state.setdefault('infra_notifications', dict())['service_id'] = infra_notifications['service_id']
            state.setdefault('infra_notifications', dict())['environment_id'] = infra_notifications['environment_id']

    def apply_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.INFO_ATTRS)
        infra_notifications = state.get('infra_notifications')
        if infra_notifications:
            template['content']['infra_notifications'] = infra_notifications
        else:
            template['content'].pop('infra_notifications', None)
        lib_common.dump_template(working_dir, lib_common.INFO_ATTRS, template)
