import copy
import shlex
import subprocess

import service_repo_client.upload_v2_controller.common as lib_common
import service_repo_client.upload_v2_controller.storages as lib_storages


class StaticFilesController(lib_common.BaseController):
    def load_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        static_files = template['content']['resources'].get('static_files', {})
        if not static_files:
            return

        static_files_storage = lib_storages.StaticFilesStorage()
        for info in static_files.itervalues():
            info['content'] = static_files_storage.get_file(info['content'])
        state.setdefault('runtime_attrs', {}).setdefault('resources', {})['static_files'] = static_files

    def apply_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        static_files = state.get('runtime_attrs', {}).get('resources', {}).get('static_files', {})
        static_files_storage = lib_storages.StaticFilesStorage()
        for info in static_files.itervalues():
            info['content'] = static_files_storage.get_file_abs_path(info['content'])
        template['content'].setdefault('resources', {})['static_files'] = static_files
        lib_common.dump_template(working_dir, lib_common.RUNTIME_ATTRS, template)


class SandboxResourcesController(lib_common.BaseController):
    def load_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        sandbox_files = template['content']['resources'].get('sandbox_files', {})
        if not sandbox_files:
            return

        sandbox_resources = dict()
        for key, info in sandbox_files.iteritems():
            sandbox_resources[key] = {
                'is_dynamic': info.get('is_dynamic', False),
                'resource_id': info['resource_id']
            }
        state.setdefault('runtime_attrs', {}).setdefault('resources', {})['sandbox_resources'] = sandbox_resources

    def update_state_before_diff(self, old_state, new_state):
        new_state = new_state.get('runtime_attrs', {}).get('resources', {}).get('sandbox_resources', {})
        old_state = old_state.get('runtime_attrs', {}).get('resources', {}).get('sandbox_resources', {})
        for key, info in new_state.iteritems():
            if info.get('disable_updates', False) and key in old_state:
                old_state[key] = copy.deepcopy(info)

    def apply_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        sandbox_resources = state.get('runtime_attrs', {}).get('resources', {}).get('sandbox_resources', {})
        sandbox_files = template['content'].setdefault('resources', {}).setdefault('sandbox_files', dict())
        sandbox_resources_storage = lib_storages.SandboxResourcesStorage()

        for key, info in sandbox_resources.iteritems():
            if info.get('disable_updates', False) and key in sandbox_files:
                continue

            sandbox_files.setdefault(key, dict()).update(sandbox_resources_storage.get_meta_info(info['resource_id']))

        for additional_file in set(sandbox_files) - set(sandbox_resources):
            sandbox_files.pop(additional_file)

        lib_common.dump_template(working_dir, lib_common.RUNTIME_ATTRS, template)


class CommandMixin(object):
    def from_template_to_state(self, template_command):
        if len(template_command) == 3 and template_command[:2] == ['/bin/sh', '-c']:
            return template_command[2]
        else:
            return template_command

    def from_state_to_template(self, state_command):
        if not isinstance(state_command, list):
            return ['/bin/sh', '-c', state_command]
        return state_command


class InitContainersController(lib_common.BaseController):
    def load_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        containers = template['content']['instance_spec'].get('initContainers', [])
        if not containers:
            return

        containers_state = dict()
        for container in containers:
            containers_state[container['name']] = CommandMixin().from_template_to_state(container['command'])
        state.setdefault('runtime_attrs', {}).setdefault('containers', {})['init'] = containers_state

    def apply_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        containers_state = state.get('runtime_attrs', {}).get('containers', {}).get('init', {})
        containers = []

        for key in sorted(containers_state.keys()):
            containers.append({
                'name': key,
                'command': CommandMixin().from_state_to_template(containers_state[key]),
            })

        template['content'].setdefault('instance_spec', {})['initContainers'] = containers
        lib_common.dump_template(working_dir, lib_common.RUNTIME_ATTRS, template)


class VolumesControllerBase(lib_common.BaseController):
    volume_type = None
    volume_template_key = None
    volume_state_key = None

    def get_volumes_from_template(self, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        return list(filter(lambda volume: volume['type'] == self.volume_type, template['content']['instance_spec'].get('volume', [])))

    def from_template_to_state(self, template_data):
        raise NotImplementedError('from_template_to_state')

    def load_state(self, state, working_dir):
        volumes = self.get_volumes_from_template(working_dir)
        if not volumes:
            return
        volumes_state = {}
        for volume in volumes:
            volumes_state[volume['name']] = self.from_template_to_state(volume[self.volume_template_key])

        state.setdefault('runtime_attrs', {}).setdefault('volumes', {})[self.volume_state_key] = volumes_state

    def from_state_to_template(self, state_data):
        raise NotImplementedError('from_state_to_template')

    def apply_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        volumes = [volume for volume in template['content']['instance_spec'].setdefault('volume', []) if volume['type'] != self.volume_type]
        volumes_state = state.get('runtime_attrs', {}).get('volumes', {}).get(self.volume_state_key, {})
        for volume_name in sorted(volumes_state.keys()):
            volumes.append({
                'name': volume_name,
                'type': self.volume_type,
                self.volume_template_key: self.from_state_to_template(volumes_state[volume_name])
            })
        template['content']['instance_spec']['volume'] = volumes
        lib_common.dump_template(working_dir, lib_common.RUNTIME_ATTRS, template)


class VaultSecretMixin(object):
    def from_template_to_state(self, template_data):
        vault_secret = template_data['vaultSecret']
        vault_secrets_storage = lib_storages.VaultSecretsStorage()
        vault_secrets_storage.add_known_secret_info(vault_secret['secretId'], vault_secret['secretName'], vault_secret['delegationToken'])
        return {
            'secret_uuid': vault_secret['secretId'],
            'version_uuid': vault_secret['secretVer'],
        }

    def from_state_to_template(self, state_data):
        vault_secrets_storage = lib_storages.VaultSecretsStorage()
        secret_info = vault_secrets_storage.get_secret_info(state_data['secret_uuid'])
        return {'vaultSecret': {
            'secretId': state_data['secret_uuid'],
            'secretVer': state_data['version_uuid'],
            'secretName': secret_info['name'],
            'delegationToken': secret_info['token']
        }}


class VaultSecretVolumesController(VolumesControllerBase, VaultSecretMixin):
    volume_type = 'VAULT_SECRET'
    volume_template_key = 'vaultSecretVolume'
    volume_state_key = 'vault_secret'

    def from_template_to_state(self, template_data):
        return VaultSecretMixin.from_template_to_state(self, template_data)

    def from_state_to_template(self, state_data):
        return VaultSecretMixin.from_state_to_template(self, state_data)


class TemplateVolumesController(VolumesControllerBase):
    volume_type = 'TEMPLATE'
    volume_template_key = 'templateVolume'
    volume_state_key = 'template'

    MAPPING = [('src', 'srcPath'), ('dst', 'dstPath')]

    def from_template_to_state(self, template_data):
        templates = template_data['template']
        return [
            lib_common.from_template_to_state_by_mapping(template, self.MAPPING)
            for template in templates
        ]

    def from_state_to_template(self, state_data):
        return {'template': [
            lib_common.from_state_to_template_by_mapping(state, self.MAPPING)
            for state in state_data
        ]}


class ItsVolumesController(VolumesControllerBase):
    volume_type = 'ITS'
    volume_template_key = 'itsVolume'
    volume_state_key = 'its'

    MAPPING = [
        ('url', 'itsUrl', 'http://its.yandex-team.ru/v1'),
        ('max_retry_polling_period_in_second', 'maxRetryPeriodSeconds', 300),
        ('polling_period_in_second', 'periodSeconds', 60),
        ('use_shared_memory', 'useSharedMemory', False),
    ]

    def from_template_to_state(self, template_data):
        return lib_common.from_template_to_state_by_mapping(template_data, self.MAPPING)

    def from_state_to_template(self, state_data):
        return lib_common.from_state_to_template_by_mapping(state_data, self.MAPPING)


class NannySecretMixin(object):
    MAPPING = [('keychain_id', 'keychainId'), ('secret_id', 'secretId'), ('revision_id', 'secretRevisionId')]

    def from_template_to_state(self, template_data):
        nanny_secret = template_data['keychainSecret']
        return lib_common.from_template_to_state_by_mapping(nanny_secret, self.MAPPING)

    def from_state_to_template(self, state_data):
        return {'keychainSecret': lib_common.from_state_to_template_by_mapping(state_data, self.MAPPING), 'secretName': ''}


class NannySecretVolumesController(VolumesControllerBase, NannySecretMixin):
    volume_type = 'SECRET'
    volume_template_key = 'secretVolume'
    volume_state_key = 'nanny_secret'

    def update_state_before_diff(self, old_state, new_state):
        new_state = new_state.get('runtime_attrs', {}).get('volumes', {}).get(self.volume_state_key, {})
        old_state = old_state.get('runtime_attrs', {}).get('volumes', {}).get(self.volume_state_key, {})
        for key in set(new_state) - set(old_state):
            raise ValueError('You try add new nanny_secret volume {}. It is deprecated. Please use vault_secret volume'.format(key))

    def from_template_to_state(self, template_data):
        return NannySecretMixin.from_template_to_state(self, template_data)

    def from_state_to_template(self, state_data):
        return NannySecretMixin.from_state_to_template(self, state_data)


class ActionHandlerMixin(object):
    def from_template_to_state(self, template_data):
        t = template_data.get('type', 'NONE')
        if t == 'NONE':
            return None
        elif t == 'EXEC':
            action = template_data['execAction']
            return {'command': CommandMixin().from_template_to_state(action['command'])}
        elif t == 'HTTP_GET':
            action = template_data['httpGet']
            result = {
                'path': action.get('path', ''),
            }
            if action.get('port'):
                result['port'] = action['port']
            return result
        else:
            assert t == 'TCP_SOCKET'
            action = template_data['tcpSocket']
            return {'port': action.get('port', '')}

    def from_state_to_template(self, state_data):
        if state_data is None:
            return {'type': 'NONE'}
        if 'command' in state_data:
            return {
                'type': 'EXEC',
                'execAction': {
                    'command': CommandMixin().from_state_to_template(state_data['command'])
                }
            }
        if 'path' in state_data:
            path = state_data.get('path')
            port = state_data.get('port')
            result = {}
            if path:
                result['path'] = path
            if port:
                result['port'] = port
            return {
                'type': 'HTTP_GET',
                'httpGet': result,
            }
        assert 'port' in state_data
        port = state_data.get('port')
        result = {}
        if port:
            result['port'] = port
        return {
            'type': 'TCP_SOCKET',
            'tcpSocket': result,
        }


class RuntimeContainersController(lib_common.BaseController):
    PROBE_MAPPING = [
        ('period_seconds', 'initialDelaySeconds', 5),
        ('backoff', 'periodBackoff', 2),
        ('max_period_seconds', 'maxPeriodSeconds', 60),
        ('min_period_seconds', 'minPeriodSeconds', 5),
    ]

    def load_command(self, template_command):
        return subprocess.list2cmdline(template_command)

    def apply_command(self, state_command):
        return shlex.split(state_command)

    def load_readiness_probe(self, template_probe):
        handlers = []
        action_handler_mixin = ActionHandlerMixin()
        for handler in template_probe.get('handlers', []):
            state_handler = action_handler_mixin.from_template_to_state(handler)
            if state_handler:
                handlers.append(state_handler)
        if not handlers:
            return
        result = {'handlers': handlers}
        result.update(lib_common.from_template_to_state_by_mapping(template_probe, self.PROBE_MAPPING))
        return result

    def apply_readiness_probe(self, state_probe):
        if not state_probe:
            return
        action_handler_mixin = ActionHandlerMixin()
        result = lib_common.from_state_to_template_by_mapping(state_probe, self.PROBE_MAPPING)
        result['handlers'] = []
        for handler in state_probe['handlers']:
            result['handlers'].append(action_handler_mixin.from_state_to_template(handler))

        return result

    RESTART_POLICY_MAPPING = [
        ('max_period_seconds', 'maxPeriodSeconds', 60),
        ('min_period_seconds', 'minPeriodSeconds', 1),
        ('backoff', 'periodBackoff', 2),
        ('jitter_seconds', 'periodJitterSeconds', 20),
        ('type', 'type', 'ALWAYS'),
    ]

    def load_restart_policy(self, template_restart_policy):
        if not template_restart_policy:
            return
        return lib_common.from_template_to_state_by_mapping(template_restart_policy, self.RESTART_POLICY_MAPPING)

    def apply_restart_policy(self, state_restart_policy):
        return lib_common.from_state_to_template_by_mapping(state_restart_policy, self.RESTART_POLICY_MAPPING)

    PRE_STOP_MAPPING = [
        ('all_handlers_barrier', 'termBarrier', 'IGNORE'),
        ('graceful_period_seconds', 'stopGracePeriodSeconds', 60),
        ('termination_period_seconds', 'terminationGracePeriodSeconds', 60),
    ]

    def load_pre_stop_handler(self, template_prestop_handler):
        action_handler_mixin = ActionHandlerMixin()
        handler = action_handler_mixin.from_template_to_state(template_prestop_handler.get('preStop', {}))
        if not handler:
            return
        result = {'handler': handler}
        result.update(lib_common.from_template_to_state_by_mapping(template_prestop_handler, self.PRE_STOP_MAPPING))
        return result

    def apply_pre_stop_handler(self, state_prestop_handler):
        if not state_prestop_handler:
            return
        action_handler_mixin = ActionHandlerMixin()
        result = lib_common.from_state_to_template_by_mapping(state_prestop_handler, self.PRE_STOP_MAPPING)
        result['preStop'] = action_handler_mixin.from_state_to_template(state_prestop_handler['handler'])
        assert result['preStop']['type'] != 'TCP_SOCKET'
        return result

    def load_env(self, template_env):
        if template_env['valueFrom']['type'] == 'LITERAL_ENV':
            result = template_env['valueFrom']['literalEnv']['value']
            field = None
        elif template_env['valueFrom']['type'] == 'VAULT_SECRET_ENV':
            vault_secret_mixin = VaultSecretMixin()
            result = vault_secret_mixin.from_template_to_state(template_env['valueFrom']['vaultSecretEnv'])
            field = template_env['valueFrom']['vaultSecretEnv'].get('field', '')
        else:
            assert template_env['valueFrom']['type'] == 'SECRET_ENV'
            nanny_secret_mixin = NannySecretMixin()
            result = nanny_secret_mixin.from_template_to_state(template_env['valueFrom']['secretEnv'])
            field = template_env['valueFrom']['secretEnv'].get('field', '')

        if field:
            result['field'] = field

        return result

    def apply_env(self, state_env):
        if not isinstance(state_env, dict):
            return {'valueFrom': {'type': 'LITERAL_ENV', 'literalEnv': {'value': state_env}}}
        else:
            if 'keychain_id' not in state_env:
                vault_secret_mixin = VaultSecretMixin()
                result = {'valueFrom': {'type': 'VAULT_SECRET_ENV', 'vaultSecretEnv': vault_secret_mixin.from_state_to_template(state_env)}}
                field_key = 'vaultSecretEnv'
            else:
                nanny_secret_mixin = NannySecretMixin()
                result = {'valueFrom': {'type': 'SECRET_ENV', 'secretEnv': nanny_secret_mixin.from_state_to_template(state_env)}}
                field_key = 'secretEnv'

            if 'field' in state_env:
                result['valueFrom'][field_key]['field'] = state_env['field']

            return result

    def load_envs(self, template):
        result = {}
        for env in template:
            result[env['name']] = self.load_env(env)
        return result

    def apply_envs(self, state):
        result = []
        for env in sorted(state.keys()):
            result.append(self.apply_env(state[env]))
            result[-1]['name'] = env
        return result

    UNISTAT_MAPPING = [
        ('path', 'path', ''),
        ('port', 'port', '{BSCONFIG_IPORT}'),
    ]

    def load_unistat(self, template_unistat):
        return [
            lib_common.from_template_to_state_by_mapping(template, self.UNISTAT_MAPPING)
            for template in template_unistat
        ]

    def apply_unistat(self, state_unistat):
        return [
            lib_common.from_state_to_template_by_mapping(state, self.UNISTAT_MAPPING)
            for state in state_unistat
        ]

    def load_reopen_log_handler(self, template_reopen_log_handler):
        action_handler_mixin = ActionHandlerMixin()
        handler = action_handler_mixin.from_template_to_state(template_reopen_log_handler.get('handler', {}))
        return handler or None

    def apply_reopen_log_handler(self, state_reopen_log_handler):
        if not state_reopen_log_handler:
            return
        action_handler_mixin = ActionHandlerMixin()
        handler = action_handler_mixin.from_state_to_template(state_reopen_log_handler)
        assert handler['type'] != 'TCP_SOCKET'
        return {'handler': handler}

    def load_one_container(self, containers_state, template_container):
        container_state = {}
        container_state['command'] = self.load_command(template_container['command'])

        readiness_probe = self.load_readiness_probe(template_container.get('readinessProbe', {}))
        if readiness_probe:
            container_state['readiness_probe'] = readiness_probe

        environment_variables = self.load_envs(template_container.get('env', []))
        if environment_variables:
            container_state['environment_variables'] = environment_variables

        restart_policy = self.load_restart_policy(template_container.get('restartPolicy'))
        if restart_policy:
            container_state['restart_policy'] = restart_policy

        pre_stop_handler = self.load_pre_stop_handler(template_container.get('lifecycle', {}))
        if pre_stop_handler:
            container_state['pre_stop_handler'] = pre_stop_handler

        unistat = self.load_unistat(template_container.get('unistatEndpoints', []))
        if unistat:
            container_state['unistat'] = unistat

        reopen_log_handler = self.load_reopen_log_handler(template_container.get('reopenLogAction', {}))
        if reopen_log_handler:
            container_state['reopen_log_handler'] = reopen_log_handler

        containers_state[template_container['name']] = container_state

    def apply_one_container(self, container_state, template_containers):
        container = {}
        container['command'] = self.apply_command(container_state['command'])

        if 'readiness_probe' in container_state:
            container['readinessProbe'] = self.apply_readiness_probe(container_state['readiness_probe'])

        if 'environment_variables' in container_state:
            container['env'] = self.apply_envs(container_state['environment_variables'])

        if 'restart_policy' in container_state:
            container['restartPolicy'] = self.apply_restart_policy(container_state['restart_policy'])

        if 'pre_stop_handler' in container_state:
            container['lifecycle'] = self.apply_pre_stop_handler(container_state['pre_stop_handler'])

        if 'unistat' in container_state:
            container['unistatEndpoints'] = self.apply_unistat(container_state['unistat'])

        if 'reopen_log_handler' in container_state:
            container['reopenLogAction'] = self.apply_reopen_log_handler(container_state['reopen_log_handler'])

        template_containers.append(container)

    def load_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        containers = template['content']['instance_spec'].get('containers', [])
        if not containers:
            return

        containers_state = dict()
        for container in containers:
            self.load_one_container(containers_state, container)
        state.setdefault('runtime_attrs', {}).setdefault('containers', {})['runtime'] = containers_state

    def apply_state(self, state, working_dir):
        template = lib_common.load_template(working_dir, lib_common.RUNTIME_ATTRS)
        containers_state = state.get('runtime_attrs', {}).get('containers', {}).get('runtime', {})
        containers = []

        for key in sorted(containers_state.keys()):
            self.apply_one_container(containers_state[key], containers)
            containers[-1]['name'] = key

        template['content'].setdefault('instance_spec', {})['containers'] = containers
        lib_common.dump_template(working_dir, lib_common.RUNTIME_ATTRS, template)
