import hashlib
import os
import shutil
import requests

from sandbox.common import rest
from sandbox.common.proxy import OAuth
from library.python.vault_client.instances import Production as VaultClient

from service_repo_client.lib.errors import AuthError


class StaticFilesStorage(object):
    @staticmethod
    def _md5(fname):
        hash_md5 = hashlib.md5()
        with open(fname, "rb") as f:
            for chunk in iter(lambda: f.read(4096), b""):
                hash_md5.update(chunk)
        return hash_md5.hexdigest()

    @staticmethod
    def _mkdir_p(path):
        if not os.path.exists(path):
            os.mkdir(path)

    STATIC_FILES = 'static_files'
    state = None

    @classmethod
    def static_file(cls, file_name):
        return os.path.join(cls.STATIC_FILES, file_name)

    @classmethod
    def load_state(cls):
        if cls.state is not None:
            return

        cls._mkdir_p(cls.STATIC_FILES)
        state = {}
        for f in os.listdir(cls.STATIC_FILES):
            state[cls._md5(cls.static_file(f))] = f

        cls.state = state

    def __init__(self):
        self.load_state()

    def get_file(self, file_path):
        md5_value = self._md5(file_path)
        if md5_value not in self.state:
            file_name = os.path.basename(file_path)
            while os.path.exists(self.static_file(file_name)):
                file_name = file_name + '.other'
            shutil.copyfile(file_path, self.static_file(file_name))

            self.state[md5_value] = file_name

        return self.state[md5_value]

    def get_file_abs_path(self, file_path):
        return os.path.abspath(self.static_file(file_path))


class SandboxResourcesStorage(object):
    SANDBOX_CLIENT = None
    SANDBOX_RESOURCES_CACHE = dict()

    @classmethod
    def init_client(cls, token):
        if not token:
            token = os.getenv('SANDBOX_OAUTH_TOKEN')
            if not token:
                msg = 'Sandbox OAuth token is not given in --sandbox_token argument and is not present in SANDBOX_OAUTH_TOKEN environment variable'
                raise AuthError(msg)
        cls.SANDBOX_CLIENT = rest.Client(auth=OAuth(token))

    def get_meta_info(self, resource_id):
        resource_id = str(resource_id)
        if resource_id not in self.SANDBOX_RESOURCES_CACHE:
            resource = self.SANDBOX_CLIENT.resource[resource_id].read()
            task = self.SANDBOX_CLIENT.task[resource['task']['id']].read()
            self.SANDBOX_RESOURCES_CACHE[resource_id] = {
                'resource_id': resource_id,
                'resource_type': resource['type'],
                'task_id': str(task['id']),
                'task_type': task['type'],
            }
        return self.SANDBOX_RESOURCES_CACHE[resource_id]


class VaultSecretsStorage(object):
    VAULT_SIGNATURE = None
    VAULT_CLIENT = None
    VAULT_SECRETS_CACHE = None

    @classmethod
    def init_client(cls, token):
        if not token:
            token = os.getenv('VAULT_TOKEN')
            if not token:
                msg = 'Vault OAuth token is not given in --vault_token argument and is not present in VAULT_TOKEN environment variable'
                raise AuthError(msg)
        cls.VAULT_CLIENT = VaultClient(authorization='OAuth {}'.format(token), decode_files=True)

    @classmethod
    def update_signature(cls, service_id):
        cls.VAULT_SIGNATURE = service_id
        cls.VAULT_SECRETS_CACHE = dict()

    def add_known_secret_info(self, secret_uuid, name, delegation_token):
        self.VAULT_SECRETS_CACHE[secret_uuid] = {'name': name, 'token': delegation_token}

    def get_secret_info(self, secret_uuid):
        if secret_uuid not in self.VAULT_SECRETS_CACHE:
            secret = self.VAULT_CLIENT.get_secret(secret_uuid)
            delegation_token, _ = self.VAULT_CLIENT.create_token(secret_uuid, tvm_client_id='2002924', signature=self.VAULT_SIGNATURE)
            self.add_known_secret_info(secret_uuid, secret['name'], delegation_token)
        return self.VAULT_SECRETS_CACHE[secret_uuid]


class GroupsStorage(object):
    staff_api_url = 'https://staff-api.yandex-team.ru/v3/groups'
    AUTH_HEADER = None
    ID_TO_URL = {}
    URL_TO_ID = {}

    @classmethod
    def init_client(cls, token):
        if not token:
            token = os.getenv('STAFF_TOKEN')
            if not token:
                msg = 'Staff OAuth token is not given in --staff_token argument and is not present in STAFF_TOKEN environment variable'
                raise AuthError(msg)
        cls.AUTH_HEADER = 'OAuth {}'.format(token)

    def _request(self, url=None, group_id=None):
        if url:
            query = 'url=="{}"'.format(url)
        else:
            assert group_id is not None
            query = 'id=={}'.format(group_id)
        response = requests.get(
            self.staff_api_url,
            params=dict(_one='1', _query=query, _fields='id,url'),
            headers={'Authorization': self.AUTH_HEADER},
            timeout=(1, 60)
        )
        response.raise_for_status()
        j = response.json()
        return str(j['id']), j['url']

    def add_known_pair(self, group_id, url):
        self.ID_TO_URL[group_id] = url
        self.URL_TO_ID[url] = group_id

    def get_id_by_url(self, url):
        if url not in self.URL_TO_ID:
            self.add_known_pair(*self._request(url=url))

        return self.URL_TO_ID[url]

    def get_url_by_id(self, group_id):
        if group_id not in self.ID_TO_URL:
            self.add_known_pair(*self._request(group_id=group_id))

        return self.ID_TO_URL[group_id]
