import argparse
import logging
import sys

from clusterpb import types_pb2
from gevent import monkey
from instancectl.clients.vault import client as vault
from instancectl.hq.volumes import secret, template, its
from instancectl.lib import confutil

monkey.patch_all()

from instancectl import hq, constants
from instancectl.lib import envutil, actionsutil


def register_args(parser):
    parser.add_argument(
        '--federated-url', type=str, default=constants.DEFAULT_FEDERATED_URL, help='federated url'
    )
    parser.add_argument(
        '--vault-url', type=str, default=constants.DEFAULT_VAULT_URL, help='vault url'
    )
    parser.add_argument(
        '--its-shared-storage', default=None, help='its shared storage'
    )
    parser.add_argument(
        '--hq-url', metavar='HQ_URL', dest='hq_url',
        help='HQ URL', type=str, default=None
    )

    parser.add_argument(
        '--its-poll', default=0, help='need poll'
    )
    parser.add_argument(
        '--its-url', default='', help='its url'
    )
    parser.add_argument(
        '--its-force-poll-timeout', default=0, help='its force poll timeout'
    )
    parser.add_argument(
        '--its-max-poll-timeout', default=0, help='its max poll timeout'
    )


def render_template_load_secrets(argv):
    def parse_args(argv):
        parser = argparse.ArgumentParser(add_help=True)

        register_args(parser)

        return parser.parse_args(argv)

    args = parse_args(argv)

    logger = logging.getLogger('render_template')

    env = envutil.make_instance_ctl_env(args.hq_url)

    hq_client = hq.instance_client.HqInstanceClient(env.hq_url)
    spec = actionsutil.get_instance_spec(hq_client, env)

    its_shared_storage = args.its_shared_storage or envutil.make_its_shared_storage_dir(env)
    vault_client = vault.VaultClient(args.vault_url, env.service_id)

    context = confutil.make_template_plugin_context(args.federated_url, env)
    secret_plugin = secret.SecretVolumePlugin(vault_client=vault_client)
    template_plugin = template.TemplateVolumePlugin(context, env.instance_dir)
    its_plugin = its.ItsVolumePlugin(
        auto_tags=env.auto_tags,
        first_poll_timeout=constants.ITS_FIRST_POLL_TIMEOUT,
        shared_storage=its_shared_storage
    )

    if args.its_poll:
        v = spec.volume.add()
        v.name = constants.ITS_CONTROLS_DIR
        v.type = types_pb2.Volume.ITS
        v.its_volume.its_url = args.its_url
        v.its_volume.period_seconds = int(args.its_force_poll_timeout)
        v.its_volume.max_retry_period_seconds = int(args.its_max_poll_timeout)

    for v in spec.volume:
        if v.type == types_pb2.Volume.SECRET:
            logger.info('downloading secrets to  %s', env.instance_dir)
            secret_plugin.setup(env.instance_dir, v)
        elif v.type == types_pb2.Volume.TEMPLATE:
            logger.info('rendering template  %s', env.instance_dir)
            template_plugin.setup(env.instance_dir, v)
        elif v.type == types_pb2.Volume.ITS:
            # We MUST NOT start instances before the first attempt of ITS polling (successful or not)
            logger.info('fetching (hq) its commands for  %s', env.instance_id)
            its_plugin.setup(env.instance_dir, v)
            its_plugin.stop()


if __name__ == '__main__':
    render_template_load_secrets(sys.argv[1:])
